/*
 * ICPL Hero YouTube Background
 * - Localized from functions.php as `ICPL_HERO_YT`.
 * - Supports start/end (seconds). If end > 0, loops between start and end.
 * - Uses YouTube IFrame Player API for reliable segment looping.
 */

(function () {
  'use strict';

  if (!window.ICPL_HERO_YT || !window.ICPL_HERO_YT.videoId) return;

  var cfg = window.ICPL_HERO_YT;
  var selector = (cfg && cfg.selector) || '.icpl-hero';
  var videoId = (cfg.videoId || '').toString().trim();
  var start = parseInt(cfg.start, 10);
  var end = parseInt(cfg.end, 10);

  if (!videoId) return;
  if (isNaN(start) || start < 0) start = 0;
  if (isNaN(end) || end < 0) end = 0;
  if (end > 0 && end <= start) end = 0;

  function ensureBgContainer(hero) {
    var bg = hero.querySelector('.icpl-hero-bg');
    if (!bg) {
      bg = document.createElement('div');
      bg.className = 'icpl-hero-bg';
      hero.insertBefore(bg, hero.firstChild);
    }

    // Avoid stacking old iframes from previous implementations
    var existingIframe = bg.querySelector('iframe');
    if (existingIframe) existingIframe.remove();

    return bg;
  }

  function loadYouTubeAPI(cb) {
    if (window.YT && window.YT.Player) {
      cb();
      return;
    }

    // If already loading, wait
    if (window.__ICPL_YT_API_LOADING__) {
      var w = setInterval(function () {
        if (window.YT && window.YT.Player) {
          clearInterval(w);
          cb();
        }
      }, 50);
      return;
    }

    window.__ICPL_YT_API_LOADING__ = true;

    var tag = document.createElement('script');
    tag.src = 'https://www.youtube.com/iframe_api';
    tag.async = true;
    document.head.appendChild(tag);

    var t = setInterval(function () {
      if (window.YT && window.YT.Player) {
        clearInterval(t);
        cb();
      }
    }, 50);
  }

  function mount() {
    var hero = document.querySelector(selector);
    if (!hero) return;

    // Only act when hero is configured for YouTube
    if (!hero.hasAttribute('data-hero-youtube')) return;

    var bg = ensureBgContainer(hero);

    // Player mount
    var mountId = 'icpl-hero-yt-player';
    var mount = bg.querySelector('#' + mountId);
    if (!mount) {
      mount = document.createElement('div');
      mount.id = mountId;
      bg.appendChild(mount);
    }

    loadYouTubeAPI(function () {
      var loopTimer = null;

      function clearLoopTimer() {
        if (loopTimer) {
          clearInterval(loopTimer);
          loopTimer = null;
        }
      }

      function startLoopWatcher(player) {
        clearLoopTimer();

        // end 未指定なら監視不要（YouTubeのloop=1で全体ループ）
        if (!(end > 0 && end > start)) return;

        // 200ms間隔で現在位置を監視して end で start に戻す
        loopTimer = setInterval(function () {
          try {
            var ct = player.getCurrentTime();
            if (typeof ct !== 'number') return;

            // end 直前で巻き戻す（誤差吸収）
            if (ct >= (end - 0.2)) {
              player.seekTo(start, true);
              player.playVideo();
            }
          } catch (e) {
            // noop
          }
        }, 200);
      }

      var player = new window.YT.Player(mountId, {
        width: '100%',
        height: '100%',
        videoId: videoId,
        playerVars: {
          autoplay: 1,
          controls: 0,
          mute: 1,
          playsinline: 1,
          loop: 1,
          playlist: videoId, // required for loop
          start: start,
          // end は playerVars にも入れる（効く場合がある）
          end: end > 0 ? end : undefined,
          rel: 0,
          modestbranding: 1,
          disablekb: 1,
          fs: 0,
          iv_load_policy: 3
        },
        events: {
          onReady: function (ev) {
            try {
              ev.target.mute();
              if (start > 0) {
                ev.target.seekTo(start, true);
              }
              ev.target.playVideo();
            } catch (e) {
              // noop
            }
            startLoopWatcher(ev.target);

            if (window.console && console.log) {
              console.log('[ICPL] hero-youtube-bg player ready:', videoId, 'start=', start, 'end=', end);
            }
          },
          onStateChange: function (ev) {
            // 0: ended
            if (ev.data === 0) {
              try {
                ev.target.seekTo(start, true);
                ev.target.playVideo();
              } catch (e) {
                // noop
              }
            }

            // 1: playing
            if (ev.data === 1) {
              startLoopWatcher(ev.target);
            }
          }
        }
      });

      window.addEventListener('beforeunload', function () {
        try {
          clearLoopTimer();
          if (player && player.destroy) player.destroy();
        } catch (e) {
          // noop
        }
      });
    });
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', mount);
  } else {
    mount();
  }
})();