<?php
/**
 * 10+ (ICPL-ichimaru-plus)
 * Theme functions.
 */

if (!defined('ABSPATH')) {
    exit;
}

define('ICPL_THEME_VERSION', '1.0.0');

/**
 * Global hero rendering flag (mutable)
 * - Avoid double output when templates also include the hero.
 */
if (!isset($GLOBALS['icpl_hero_rendered'])) {
    $GLOBALS['icpl_hero_rendered'] = false;
}

/**
 * Decide which post/page ID should be used as the hero source for the current request.
 * This makes hero rendering template-agnostic.
 */
function icpl_get_hero_source_post_id() {
    // Singular (page/post/custom post)
    if (is_singular()) {
        $id = (int) get_queried_object_id();
        return ($id > 0) ? $id : 0;
    }

    // Blog home (posts index) and archives/search: use the "Posts page" if set
    if (is_home() || is_archive() || is_search()) {
        $posts_page = (int) get_option('page_for_posts');
        return ($posts_page > 0) ? $posts_page : 0;
    }

    // Front page (static page)
    if (is_front_page()) {
        $front = (int) get_option('page_on_front');
        return ($front > 0) ? $front : 0;
    }

    return 0;
}

/**
 * Render the theme hero (collage / featured image) without relying on individual templates.
 * - Outputs template-parts/page-hero if it exists.
 * - Works for: singular, blog home, archives, search (via page_for_posts), front page.
 */
function icpl_maybe_render_global_hero() {
    // Only front-end
    if (is_admin() || wp_doing_ajax()) {
        return;
    }


    // Avoid double render (template may already have rendered it)
    if (!empty($GLOBALS['icpl_hero_rendered'])) {
        return;
    }

    $post_id = (int) icpl_get_hero_source_post_id();
    if ($post_id <= 0) {
        return;
    }

    // 投稿タイプがカスタマイザーで許可されていない場合はヒーローを表示しない
    $pt = get_post_type($post_id);
    if (!is_string($pt) || $pt === '') {
        return;
    }
    if (!icpl_is_hero_enabled_for_post_type($pt)) {
        return;
    }

    // Render rule: collage enabled OR featured image exists
    $icpl_collage_enable = get_post_meta($post_id, 'icpl_hero_collage_enable', true);
    $icpl_use_hero = ((string) $icpl_collage_enable === '1') || has_post_thumbnail($post_id);
    if (!$icpl_use_hero) {
        return;
    }

    // Temporarily switch global $post so page-hero.php can use get_the_ID()/post meta safely.
    global $post;
    $icpl__orig_post = $post;
    $icpl__hero_post = get_post($post_id);

    if ($icpl__hero_post instanceof WP_Post) {
        $post = $icpl__hero_post;
        setup_postdata($post);
    }

    // Mark rendered
    $GLOBALS['icpl_hero_rendered'] = true;

    if (locate_template('template-parts/page-hero.php')) {
        get_template_part('template-parts/page-hero');
    }

    if ($icpl__hero_post instanceof WP_Post) {
        wp_reset_postdata();
        $post = $icpl__orig_post;
    }
}

/**
 * Latest News (front parts)
 * - lives in /includes/latest-news.php
 * - loaded early so Customizer settings/controls are registered reliably
 */
$icpl_latest_news_inc = get_template_directory() . '/includes/latest-news.php';
if (file_exists($icpl_latest_news_inc)) {
    require_once $icpl_latest_news_inc;
}

/**
 * Page Hero (per-page collage metabox)
 * - lives in /includes/metaboxes-page-hero.php
 */
$icpl_page_hero_metabox_inc = get_template_directory() . '/includes/metaboxes-page-hero.php';
if (file_exists($icpl_page_hero_metabox_inc)) {
    require_once $icpl_page_hero_metabox_inc;
}

/**
 * Typography (Japanese fonts)
 * - lives in /includes/typography-fonts.php
 */
$icpl_typography_inc = get_template_directory() . '/includes/typography-fonts.php';
if (file_exists($icpl_typography_inc)) {
    require_once $icpl_typography_inc;
}

/**
 * Banner Widget (Page+ style)
 * - lives in /includes/banner-icpl-ichimaru-plus.php
 */
$icpl_banner_widget_inc = get_template_directory() . '/includes/banner-icpl-ichimaru-plus.php';
if (file_exists($icpl_banner_widget_inc)) {
    require_once $icpl_banner_widget_inc;
}

function icpl_theme_setup() {
    load_theme_textdomain('icpl-ichimaru-plus', get_template_directory() . '/languages');

    add_theme_support('title-tag');
    add_theme_support('post-thumbnails');
    add_theme_support('automatic-feed-links');
    add_theme_support('html5', array(
        'search-form','comment-form','comment-list','gallery','caption','style','script'
    ));
    add_theme_support('custom-logo', array(
        'height'      => 80,
        'width'       => 240,
        'flex-height' => true,
        'flex-width'  => true,
    ));
    add_theme_support('customize-selective-refresh-widgets');
    add_theme_support('wp-block-styles');
    add_theme_support('align-wide');
    add_theme_support('responsive-embeds');

    register_nav_menus(array(
        'primary' => __('Primary Menu', 'icpl-ichimaru-plus'),
        'footer'  => __('Footer Menu', 'icpl-ichimaru-plus'),
    ));

    // Editor font sizes / palette (minimal)
    add_theme_support('editor-font-sizes', array(
        array(
            'name' => __('Small', 'icpl-ichimaru-plus'),
            'size' => 12,
            'slug' => 'small',
        ),
        array(
            'name' => __('Normal', 'icpl-ichimaru-plus'),
            'size' => 16,
            'slug' => 'normal',
        ),
        array(
            'name' => __('Large', 'icpl-ichimaru-plus'),
            'size' => 24,
            'slug' => 'large',
        ),
    ));
}
add_action('after_setup_theme', 'icpl_theme_setup');

function icpl_enqueue_assets() {
    wp_enqueue_style('icpl-ichimaru-plus-style', get_stylesheet_uri(), array(), ICPL_THEME_VERSION);
}
add_action('wp_enqueue_scripts', 'icpl_enqueue_assets');

/**
 * ヒーロー表示可否（投稿タイプ別）
 * - カスタマイザーの設定キー：
 *   icpl_page_hero_apply_{post_type}
 * - デフォルトは「固定ページのみON」
 */
function icpl_is_hero_enabled_for_post_type(string $post_type): bool {
    $default = ($post_type === 'page');
    $raw = get_theme_mod('icpl_page_hero_apply_' . $post_type, $default ? 1 : 0);

    // 過去データとの互換のため true も許可するが、
    // 原則「文字列の '1' のときのみON」とする
    return ($raw === true || (string) $raw === '1');
}

// Template-agnostic hero injection (render right AFTER the site header).
// - header.php will call do_action('icpl_after_header') once per request.
if (!has_action('icpl_after_header', 'icpl_maybe_render_global_hero')) {
    add_action('icpl_after_header', 'icpl_maybe_render_global_hero', 1);
}

/**
 * Breadcrumbs (simple, dependency-free)
 * - Rendered via icpl_after_header so header.php does not need to be edited.
 * - Hidden on front page by default.
 */
function icpl_render_breadcrumbs() {

    // Front-end only
    if (is_admin() || wp_doing_ajax()) {
        return;
    }

    // Do not show on front page
    if (is_front_page()) {
        return;
    }

    // If 404, skip
    if (is_404()) {
        return;
    }

    $items = array();

    // Home
    $items[] = array(
        'label' => __('ホーム', 'icpl-ichimaru-plus'),
        'url'   => home_url('/'),
    );

    // Blog posts index (if set) for posts, archives, and single posts of "post"
    $posts_page_id = (int) get_option('page_for_posts');
    $is_post_context = (is_home() || is_archive() || is_search() || (is_singular('post')));
    if ($is_post_context && $posts_page_id > 0) {
        $items[] = array(
            'label' => get_the_title($posts_page_id),
            'url'   => get_permalink($posts_page_id),
        );
    }

    // カスタム投稿タイプのアーカイブ
    // - 例: /codeplus/ のような一覧
    if (is_post_type_archive()) {
        $pt_obj = get_queried_object();
        if ($pt_obj && !is_wp_error($pt_obj) && !empty($pt_obj->labels->name)) {
            $items[] = array(
                'label' => (string) $pt_obj->labels->name,
                'url'   => '',
            );
        }
    }
    // タクソノミーアーカイブ：ターム名を末尾に追加（リンクなし）
    elseif (is_category() || is_tag() || is_tax()) {
        $term = get_queried_object();
        if ($term && !is_wp_error($term)) {
            $items[] = array(
                'label' => (string) single_term_title('', false),
                'url'   => '',
            );
        }
    }
    // 検索結果
    elseif (is_search()) {
        $items[] = array(
            'label' => sprintf(__('検索: %s', 'icpl-ichimaru-plus'), get_search_query()),
            'url'   => '',
        );
    }
    // Date archives
    elseif (is_date()) {
        if (is_year()) {
            $items[] = array('label' => get_the_date('Y'), 'url' => '');
        } elseif (is_month()) {
            $items[] = array('label' => get_the_date('Y年n月'), 'url' => '');
        } elseif (is_day()) {
            $items[] = array('label' => get_the_date('Y年n月j日'), 'url' => '');
        }
    }
    // Author archive
    elseif (is_author()) {
        $items[] = array(
            'label' => (string) get_the_author(),
            'url'   => '',
        );
    }
    // Singular (page/post/custom post): include ancestors and current title
    elseif (is_singular()) {
        $post_id = (int) get_queried_object_id();

        // For pages: ancestors
        if (is_page($post_id)) {
            $ancestors = array_reverse(get_post_ancestors($post_id));
            foreach ($ancestors as $aid) {
                $items[] = array(
                    'label' => get_the_title($aid),
                    'url'   => get_permalink($aid),
                );
            }
        }

        // For post: try category
        if (is_singular('post')) {
            $cats = get_the_category($post_id);
            if (!empty($cats) && !is_wp_error($cats)) {
                // Use primary-ish: first category
                $cat = $cats[0];
                $items[] = array(
                    'label' => $cat->name,
                    'url'   => get_category_link($cat->term_id),
                );
            }
        }

        // Current
        $items[] = array(
            'label' => get_the_title($post_id),
            'url'   => '',
        );
    }

    // If only Home, do not render
    if (count($items) <= 1) {
        return;
    }

    // Mark rendered (used to avoid duplicates / enable fallback)
    $GLOBALS['icpl_breadcrumbs_rendered'] = true;

    echo '<nav class="icpl-breadcrumb" aria-label="' . esc_attr__('パンくずリスト', 'icpl-ichimaru-plus') . '">';
    echo '<ol class="icpl-breadcrumb__list">';

    $last_index = count($items) - 1;
    foreach ($items as $i => $it) {
        $label = isset($it['label']) ? (string) $it['label'] : '';
        $url   = isset($it['url']) ? (string) $it['url'] : '';

        echo '<li class="icpl-breadcrumb__item">';
        if ($url !== '' && $i !== $last_index) {
            echo '<a class="icpl-breadcrumb__link" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
        } else {
            echo '<span class="icpl-breadcrumb__current" aria-current="page">' . esc_html($label) . '</span>';
        }
        echo '</li>';
    }

    echo '</ol>';
    echo '</nav>';
}

/**
 * パンくずリストの表示位置：ヒーローの下
 * - header.php の do_action('icpl_after_header') で出力する
 * - ヒーローより後に表示したいので優先度は 2（ヒーローは 1）
 */
if (!has_action('icpl_after_header', 'icpl_render_breadcrumbs')) {
    add_action('icpl_after_header', 'icpl_render_breadcrumbs', 2);
}


/**
 * 以前の「フッター直前へ移動する」方式は無効化
 * - 現在はヘッダー直下（最上部）に表示するため不要
 * - レイアウトの二重表示や JS での移動を避ける
 */

function icpl_register_sidebars() {
    register_sidebar(array(
        'name'          => __('Sidebar', 'icpl-ichimaru-plus'),
        'id'            => 'sidebar-1',
        'description'   => __('Main sidebar widgets.', 'icpl-ichimaru-plus'),
        'before_widget' => '<section id="%1$s" class="icpl-widget widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h2 class="widget-title">',
        'after_title'   => '</h2>',
    ));

    register_sidebar(array(
        'name'          => __('Footer Widgets', 'icpl-ichimaru-plus'),
        'id'            => 'footer-1',
        'description'   => __('Footer widget area.', 'icpl-ichimaru-plus'),
        'before_widget' => '<section id="%1$s" class="icpl-widget widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h2 class="widget-title">',
        'after_title'   => '</h2>',
    ));
}
add_action('widgets_init', 'icpl_register_sidebars');

/**
 * Customizer: hero copy and CTA for front page.
 */
function icpl_customize_register($wp_customize) {
    $wp_customize->add_section('icpl_home', array(
        'title'    => __('ホーム（フロントページ）', 'icpl-ichimaru-plus'),
        'priority' => 30,
    ));

    $wp_customize->add_setting('icpl_hero_title', array(
        'default'           => __('仕事とブランドを、きちんと伝える。', 'icpl-ichimaru-plus'),
        'sanitize_callback' => 'sanitize_text_field',
    ));
    $wp_customize->add_control('icpl_hero_title', array(
        'label'   => __('ヒーロータイトル', 'icpl-ichimaru-plus'),
        'section' => 'icpl_home',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('icpl_hero_text', array(
        'default'           => __('軽量・堅牢・拡張しやすい。10+ は、ビジネスサイトに必要な王道レイアウトを最小構成で提供します。', 'icpl-ichimaru-plus'),
        'sanitize_callback' => 'sanitize_textarea_field',
    ));
    $wp_customize->add_control('icpl_hero_text', array(
        'label'   => __('ヒーローテキスト', 'icpl-ichimaru-plus'),
        'section' => 'icpl_home',
        'type'    => 'textarea',
    ));

    $wp_customize->add_setting('icpl_hero_cta_label', array(
        'default'           => __('お問い合わせ', 'icpl-ichimaru-plus'),
        'sanitize_callback' => 'sanitize_text_field',
    ));
    $wp_customize->add_control('icpl_hero_cta_label', array(
        'label'   => __('ボタン文言', 'icpl-ichimaru-plus'),
        'section' => 'icpl_home',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('icpl_hero_cta_url', array(
        'default'           => home_url('/contact/'),
        'sanitize_callback' => 'esc_url_raw',
    ));
    $wp_customize->add_control('icpl_hero_cta_url', array(
        'label'   => __('ボタンURL', 'icpl-ichimaru-plus'),
        'section' => 'icpl_home',
        'type'    => 'url',
    ));

    // ==============================
    // Hero Background (Image / YouTube)
    // ==============================

    $wp_customize->add_setting('icpl_hero_bg_type', array(
        'default'           => 'none',
        'sanitize_callback' => function ($v) {
            $allowed = array('none', 'image', 'youtube');
            return in_array($v, $allowed, true) ? $v : 'none';
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_hero_bg_type', array(
        'label'       => __('ヒーロー背景タイプ', 'icpl-ichimaru-plus'),
        'description' => __('ヒーロー背景に「画像」または「YouTube動画」を設定できます。', 'icpl-ichimaru-plus'),
        'section'     => 'icpl_home',
        'type'        => 'radio',
        'choices'     => array(
            'none'    => __('なし', 'icpl-ichimaru-plus'),
            'image'   => __('画像', 'icpl-ichimaru-plus'),
            'youtube' => __('YouTube動画', 'icpl-ichimaru-plus'),
        ),
        'priority'    => 60,
    ));

    // Image background
    $wp_customize->add_setting('icpl_hero_bg_image', array(
        'default'           => 0,
        'sanitize_callback' => 'absint',
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control(new WP_Customize_Media_Control(
        $wp_customize,
        'icpl_hero_bg_image',
        array(
            'label'       => __('ヒーロー背景画像', 'icpl-ichimaru-plus'),
            'description' => __('背景タイプが「画像」の場合に使用します。', 'icpl-ichimaru-plus'),
            'section'     => 'icpl_home',
            'mime_type'   => 'image',
            'priority'    => 61,
            'active_callback' => function () { return get_theme_mod('icpl_hero_bg_type', 'none') === 'image'; },
        )
    ));

    // YouTube background URL
    $wp_customize->add_setting('icpl_hero_bg_youtube', array(
        'default'           => '',
        'sanitize_callback' => 'esc_url_raw',
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_hero_bg_youtube', array(
        'label'       => __('ヒーロー背景 YouTube URL', 'icpl-ichimaru-plus'),
        'description' => __('背景タイプが「YouTube動画」の場合に使用します。例： https://www.youtube.com/watch?v=VIDEO_ID', 'icpl-ichimaru-plus'),
        'section'     => 'icpl_home',
        'type'        => 'url',
        'priority'    => 62,
        'active_callback' => function () { return get_theme_mod('icpl_hero_bg_type', 'none') === 'youtube'; },
    ));

    // YouTube 再生開始位置（秒）
    $wp_customize->add_setting('icpl_hero_bg_youtube_start', array(
        'default'           => 0,
        'sanitize_callback' => function ($v) {
            $v = absint($v);
            if ($v < 0) { $v = 0; }
            if ($v > 86400) { $v = 86400; } // 最大24時間
            return $v;
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_hero_bg_youtube_start', array(
        'label'       => __('YouTube 再生開始位置（秒）', 'icpl-ichimaru-plus'),
        'description' => __('0 の場合は先頭から再生します。例：30 と入力すると30秒から開始します。', 'icpl-ichimaru-plus'),
        'section'     => 'icpl_home',
        'type'        => 'number',
        'input_attrs' => array(
            'min'  => 0,
            'step' => 1,
        ),
        'priority'    => 625,
        'active_callback' => function () { return get_theme_mod('icpl_hero_bg_type', 'none') === 'youtube'; },
    ));

    // YouTube 再生終了位置（秒）
    // 0 の場合は「終了指定なし（動画の最後まで）」として扱う
    $wp_customize->add_setting('icpl_hero_bg_youtube_end', array(
        'default'           => 0,
        'sanitize_callback' => function ($v) {
            $v = absint($v);
            if ($v < 0) { $v = 0; }
            if ($v > 86400) { $v = 86400; } // 最大24時間
            return $v;
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_hero_bg_youtube_end', array(
        'label'       => __('YouTube 再生終了位置（秒）', 'icpl-ichimaru-plus'),
        'description' => __('0 の場合は終了指定なし（動画の最後まで）です。開始〜終了の範囲でループさせたい場合に設定してください。', 'icpl-ichimaru-plus'),
        'section'     => 'icpl_home',
        'type'        => 'number',
        'input_attrs' => array(
            'min'  => 0,
            'step' => 1,
        ),
        'priority'    => 626,
        'active_callback' => function () { return get_theme_mod('icpl_hero_bg_type', 'none') === 'youtube'; },
    ));

    // Overlay darkness for readability
    $wp_customize->add_setting('icpl_hero_bg_overlay', array(
        'default'           => 35,
        'sanitize_callback' => function ($v) {
            $v = absint($v);
            if ($v < 0) { $v = 0; }
            if ($v > 90) { $v = 90; }
            return $v;
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_hero_bg_overlay', array(
        'label'       => __('ヒーロー背景の暗さ（オーバーレイ %）', 'icpl-ichimaru-plus'),
        'description' => __('文字が読みやすいように暗い幕をかけます（0〜90）。', 'icpl-ichimaru-plus'),
        'section'     => 'icpl_home',
        'type'        => 'range',
        'input_attrs' => array(
            'min'  => 0,
            'max'  => 90,
            'step' => 1,
        ),
        'priority'    => 630,
        'active_callback' => function () { return get_theme_mod('icpl_hero_bg_type', 'none') === 'youtube'; },
    ));

    // Hero height (vh)
    $wp_customize->add_setting('icpl_hero_height_vh', array(
        'default'           => 70,
        'sanitize_callback' => function ($v) {
            $v = absint($v);
            if ($v < 30) { $v = 30; }
            if ($v > 100) { $v = 100; }
            return $v;
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_hero_height_vh', array(
        'label'       => __('ヒーローの高さ（画面に対する割合）', 'icpl-ichimaru-plus'),
        'description' => __('ヒーローの最小高さを vh で指定します（30〜100）。例：70 = 画面高さの70%。', 'icpl-ichimaru-plus'),
        'section'     => 'icpl_home',
        'type'        => 'range',
        'input_attrs' => array(
            'min'  => 30,
            'max'  => 100,
            'step' => 1,
        ),
        'priority'    => 640,
    ));
}
add_action('customize_register', 'icpl_customize_register');

/**
 * Customizer: 「サイト基本情報（Site Identity）」の末尾に「カラー設定」ボタンを追加し、
 * 押下で 10+ のカラー設定セクションへ移動できるようにする。
 */
add_action('customize_register', function ($wp_customize) {

    // ボタン用カスタムコントロール
    if (!class_exists('ICPL_Customize_Section_Link_Control')) {
        class ICPL_Customize_Section_Link_Control extends WP_Customize_Control {
            public $type = 'icpl_section_link';
            public $target_section = '';

            public function render_content() {
                if (empty($this->label)) {
                    return;
                }
                ?>
                <span class="customize-control-title"><?php echo esc_html($this->label); ?></span>
                <?php if (!empty($this->description)) : ?>
                    <p class="description"><?php echo esc_html($this->description); ?></p>
                <?php endif; ?>
                <button type="button"
                        class="button button-secondary icpl-open-section"
                        data-target-section="<?php echo esc_attr($this->target_section); ?>">
                    <?php echo esc_html__('カラー/フォントを開く', 'icpl-ichimaru-plus'); ?>
                </button>
                <?php
            }
        }
    }

    // Site Identity（title_tagline）に「カラー設定」ボタンを追加
    $wp_customize->add_setting('icpl_open_colors_dummy', array(
        'sanitize_callback' => '__return_empty_string',
    ));

    $wp_customize->add_control(new ICPL_Customize_Section_Link_Control(
        $wp_customize,
        'icpl_open_colors_dummy',
        array(
            'label'          => __('カラー/フォント', 'icpl-ichimaru-plus'),
            'description'    => __('背景（白/黒）・テーマ色（差し色）・フォントを設定します。', 'icpl-ichimaru-plus'),
            'section'        => 'title_tagline',
            'priority'       => 999,
            'target_section' => 'icpl_colors',
        )
    ));

    // 10+ カラー設定セクション
    $wp_customize->add_section('icpl_colors', array(
        'title'       => __('カラー/フォント', 'icpl-ichimaru-plus'),
        'priority'    => 31,
        'description' => __('背景（白/黒）とテーマ色（差し色）、フォントを設定します。', 'icpl-ichimaru-plus'),
    ));

    // 背景モード（白/黒）
    $wp_customize->add_setting('icpl_color_mode', array(
        'default'           => 'light',
        'sanitize_callback' => function ($v) {
            return in_array($v, array('light', 'dark'), true) ? $v : 'light';
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_color_mode', array(
        'label'   => __('基本背景モード', 'icpl-ichimaru-plus'),
        'section' => 'icpl_colors',
        'type'    => 'radio',
        'choices' => array(
            'light' => __('白（Light）', 'icpl-ichimaru-plus'),
            'dark'  => __('黒（Dark）', 'icpl-ichimaru-plus'),
        ),
    ));

    // テーマ色（差し色）
    $wp_customize->add_setting('icpl_accent_color', array(
        'default'           => '#2563eb',
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control(new WP_Customize_Color_Control(
        $wp_customize,
        'icpl_accent_color',
        array(
            'label'   => __('テーマ色（差し色）', 'icpl-ichimaru-plus'),
            'section' => 'icpl_colors',
        )
    ));

    // ヘッダー背景色（任意：未設定なら差し色を使用）
    $wp_customize->add_setting('icpl_header_bg_color', array(
        'default'           => '',
        'sanitize_callback' => function ($v) {
            $v = trim((string) $v);
            if ($v === '') { return ''; }
            return sanitize_hex_color($v) ?: '';
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control(new WP_Customize_Color_Control(
        $wp_customize,
        'icpl_header_bg_color',
        array(
            'label'       => __('ヘッダー背景色（任意）', 'icpl-ichimaru-plus'),
            'description' => __('未設定の場合は「テーマ色（差し色）」が使用されます。', 'icpl-ichimaru-plus'),
            'section'     => 'icpl_colors',
        )
    ));

    // フッター背景色（任意：未設定なら差し色を使用）
    $wp_customize->add_setting('icpl_footer_bg_color', array(
        'default'           => '',
        'sanitize_callback' => function ($v) {
            $v = trim((string) $v);
            if ($v === '') { return ''; }
            return sanitize_hex_color($v) ?: '';
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control(new WP_Customize_Color_Control(
        $wp_customize,
        'icpl_footer_bg_color',
        array(
            'label'       => __('フッター背景色（任意）', 'icpl-ichimaru-plus'),
            'description' => __('未設定の場合は「テーマ色（差し色）」が使用されます。', 'icpl-ichimaru-plus'),
            'section'     => 'icpl_colors',
        )
    ));

    // WordPress標準の「サイトタイトルとキャッチフレーズを表示」(display_header_text) は
    // 10+ では混乱しやすいので非表示にする（表示制御は下の個別チェックで行う）
    if (method_exists($wp_customize, 'remove_control')) {
        $wp_customize->remove_control('display_header_text');
    }

    // ロゴ表示サイズ（高さ）
    $wp_customize->add_setting('icpl_logo_max_height', array(
        'default'           => 44,
        'sanitize_callback' => function ($v) {
            $v = absint($v);
            if ($v < 16) { $v = 16; }
            if ($v > 120) { $v = 120; }
            return $v;
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_logo_max_height', array(
        'label'       => __('ロゴ表示サイズ（高さ）', 'icpl-ichimaru-plus'),
        'description' => __('ヘッダー内でのロゴ画像の最大高さ（px）を指定します。', 'icpl-ichimaru-plus'),
        'section'     => 'title_tagline',
        'type'        => 'range',
        'input_attrs' => array(
            'min'  => 16,
            'max'  => 120,
            'step' => 1,
        ),
        'priority'    => 11,
    ));

    // サイトタイトル表示（ブログ名入力の近く）
    $wp_customize->add_setting('icpl_show_site_title', array(
        'default'           => 1,
        'sanitize_callback' => function ($v) { return $v ? 1 : 0; },
        'transport'         => 'refresh',
    ));
    $wp_customize->add_control('icpl_show_site_title', array(
        'label'    => __('サイトタイトルを表示', 'icpl-ichimaru-plus'),
        'section'  => 'title_tagline',
        'type'     => 'checkbox',
        'priority' => 21,
    ));

    // キャッチフレーズ表示（キャッチフレーズ入力の近く）
    $wp_customize->add_setting('icpl_show_tagline', array(
        'default'           => 1,
        'sanitize_callback' => function ($v) { return $v ? 1 : 0; },
        'transport'         => 'refresh',
    ));
    $wp_customize->add_control('icpl_show_tagline', array(
        'label'    => __('キャッチフレーズを表示', 'icpl-ichimaru-plus'),
        'section'  => 'title_tagline',
        'type'     => 'checkbox',
        'priority' => 23,
    ));
});

/**
 * Customizer Controls: Site Identityのボタン押下で対象セクションを開く（外部JS不要のインライン）。
 */
add_action('customize_controls_enqueue_scripts', function () {
    wp_add_inline_script(
        'customize-controls',
        "(function(api){api.bind('ready',function(){document.addEventListener('click',function(e){var btn=e.target.closest('.icpl-open-section');if(!btn){return;}var target=btn.getAttribute('data-target-section');if(!target||!api.section(target)){return;}api.section(target).focus();});});})(wp.customize);"
    );
});

/**
 * フロント側にCSS変数として反映（白/黒 + 差し色）
 */
add_action('wp_enqueue_scripts', function () {
    $mode   = get_theme_mod('icpl_color_mode', 'light');
    $accent = get_theme_mod('icpl_accent_color', '#2563eb');
    $accent = sanitize_hex_color($accent) ?: '#2563eb';

    // 背景色が暗いかどうかをざっくり判定（ヘッダー/フッターの文字色自動決定用）
    $icpl_is_dark_hex = function ($hex) {
        $hex = ltrim((string) $hex, '#');
        if (strlen($hex) === 3) {
            $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
        }
        if (strlen($hex) !== 6) {
            return true;
        }
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        // perceived luminance (0..255)
        $lum = (0.299 * $r) + (0.587 * $g) + (0.114 * $b);
        return $lum < 140;
    };

    $logo_h = (int) get_theme_mod('icpl_logo_max_height', 44);
    if ($logo_h < 16) { $logo_h = 16; }
    if ($logo_h > 120) { $logo_h = 120; }

    $show_title   = (int) get_theme_mod('icpl_show_site_title', 1);
    $show_tagline = (int) get_theme_mod('icpl_show_tagline', 1);

    // ロゴが無いのにタイトルも非表示だと何も残らないため、この場合はタイトルを強制表示する。
    $has_logo = (function_exists('has_custom_logo') && has_custom_logo());
    if (!$has_logo && $show_title !== 1) {
        $show_title = 1;
    }

    // ベース（白/黒）は固定：破綻防止
    if ($mode === 'dark') {
        $bg      = '#0b0f19';
        $text    = '#e5e7eb';
        $muted   = '#9ca3af';
        $border  = '#1f2937';
        $surface = '#111827';
    } else {
        $bg      = '#ffffff';
        $text    = '#111827';
        $muted   = '#6b7280';
        $border  = '#e5e7eb';
        $surface = '#f8fafc';
    }

    // ヘッダー/フッターの背景色（未設定なら差し色）
    $header_bg = trim((string) get_theme_mod('icpl_header_bg_color', ''));
    $footer_bg = trim((string) get_theme_mod('icpl_footer_bg_color', ''));

    $header_bg = ($header_bg !== '') ? (sanitize_hex_color($header_bg) ?: '') : '';
    $footer_bg = ($footer_bg !== '') ? (sanitize_hex_color($footer_bg) ?: '') : '';

    if ($header_bg === '') { $header_bg = $accent; }
    if ($footer_bg === '') { $footer_bg = $accent; }

    $header_text = $icpl_is_dark_hex($header_bg) ? '#ffffff' : '#111827';
    $footer_text = $icpl_is_dark_hex($footer_bg) ? '#ffffff' : '#111827';

    // リンク色は文字色に寄せる（ヘッダー/フッター内）
    $header_link = ($header_text === '#ffffff') ? 'rgba(255,255,255,.92)' : 'rgba(17,24,39,.92)';
    $footer_link = ($footer_text === '#ffffff') ? 'rgba(255,255,255,.92)' : 'rgba(17,24,39,.92)';

    // hover は差し色から自動生成（設定項目を増やさない）
    $accent_hover = icpl_adjust_hex($accent, ($mode === 'dark') ? 12 : -10);

    $css = ':root{'
        . '--icpl-bg:' . $bg . ';'
        . '--icpl-text:' . $text . ';'
        . '--icpl-muted:' . $muted . ';'
        . '--icpl-border:' . $border . ';'
        . '--icpl-surface:' . $surface . ';'
        . '--icpl-accent:' . $accent . ';'
        . '--icpl-accent-2:' . $accent_hover . ';'
        . '--icpl-header-bg:' . $header_bg . ';'
        . '--icpl-header-text:' . $header_text . ';'
        . '--icpl-footer-bg:' . $footer_bg . ';'
        . '--icpl-footer-text:' . $footer_text . ';'
        . '}'
        // base links + primary buttons
        . 'a{color:var(--icpl-accent);} '
        . '.icpl-btn-primary{background:var(--icpl-accent);border-color:var(--icpl-accent);} '
        . '.icpl-btn-primary:hover{background:var(--icpl-accent-2);border-color:var(--icpl-accent-2);} '
        // header/footer colors (use dedicated bg if set; otherwise accent)
        . '.icpl-header,header.site-header{background:var(--icpl-header-bg) !important;color:var(--icpl-header-text) !important;} '
        . '.icpl-footer,footer.site-footer{background:var(--icpl-footer-bg) !important;color:var(--icpl-footer-text) !important;} '
        . '.icpl-header a,header.site-header a{color:' . $header_link . ' !important;} '
        . '.icpl-footer a,footer.site-footer a{color:' . $footer_link . ' !important;} '
        . '.icpl-header a:hover,header.site-header a:hover{color:var(--icpl-header-text) !important;text-decoration:none;} '
        . '.icpl-footer a:hover,footer.site-footer a:hover{color:var(--icpl-footer-text) !important;text-decoration:none;} '
        . '.icpl-header .icpl-site-desc,header.site-header .icpl-site-desc{color:inherit !important;opacity:.80;} '
        // custom logo sizing (force visible)
        . '.icpl-branding{display:flex !important;align-items:center !important;} '
        . '.custom-logo-link,.icpl-custom-logo-link{display:inline-flex !important;align-items:center !important;line-height:1 !important;margin-right:10px !important;visibility:visible !important;} '
        . '.custom-logo,.icpl-custom-logo{display:block !important;visibility:visible !important;height:auto !important;width:auto !important;max-height:' . $logo_h . 'px !important;} '
        // パンくず（帯スタイル・区切りは >）
        . '.icpl-breadcrumb{width:100%;margin:0;padding:3px 0;background:var(--icpl-surface);border-bottom:1px solid var(--icpl-border);} '
        . '.icpl-breadcrumb__list{list-style:none;margin:0 auto;max-width:var(--wp--style--global--content-size, 1200px);display:flex;flex-wrap:wrap;align-items:center;font-size:13px;line-height:1.4;} '
        . '.icpl-breadcrumb__item{display:flex;align-items:center;} '
        . '.icpl-breadcrumb__item:after{content:" > ";display:inline-block !important;padding:0 8px;color:var(--icpl-muted);} '
        . '.icpl-breadcrumb__item:last-child:after{content:"";display:none !important;padding:0;} '
        . '.icpl-breadcrumb__item:before{content:"";display:none !important;} '
        . '.icpl-breadcrumb__link{color:var(--icpl-muted);text-decoration:none;} '
        . '.icpl-breadcrumb__link:hover{text-decoration:underline;} '
        . '.icpl-breadcrumb__current{color:var(--icpl-text);opacity:.9;} ';

    // ==============================
    // Hero background (image / YouTube)
    // ==============================
    $hero_h_vh = (int) get_theme_mod('icpl_hero_height_vh', 70);
    if ($hero_h_vh < 30) { $hero_h_vh = 30; }
    if ($hero_h_vh > 100) { $hero_h_vh = 100; }

    $overlay = (int) get_theme_mod('icpl_hero_bg_overlay', 35);
    if ($overlay < 0) { $overlay = 0; }
    if ($overlay > 90) { $overlay = 90; }
    $overlay_alpha = $overlay / 100;

    $css .= '.icpl-hero{position:relative;overflow:hidden;min-height:' . $hero_h_vh . 'vh;display:flex;align-items:center;} '
          . '.icpl-hero > .icpl-hero-bg{position:absolute;inset:0;z-index:0;} '
          . '.icpl-hero .icpl-hero-inner{position:relative;z-index:2;} '
          . '.icpl-hero:before{content:"";position:absolute;inset:0;z-index:1;background:rgba(0,0,0,' . $overlay_alpha . ');} '
          . '.icpl-hero-bg iframe{position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);width:177.77vh;height:100vh;min-width:100%;min-height:56.25vw;pointer-events:none;} '
          . '@media (prefers-reduced-motion: reduce){.icpl-hero-bg iframe{display:none !important;}} ';

    // If image background selected, apply as CSS background-image
    $bg_type = (string) get_theme_mod('icpl_hero_bg_type', 'none');
    if ($bg_type === 'image') {
        $img_id = (int) get_theme_mod('icpl_hero_bg_image', 0);
        if ($img_id > 0) {
            $img_url = wp_get_attachment_image_url($img_id, 'full');
            if ($img_url) {
                $css .= '.icpl-hero{background-image:url(' . esc_url($img_url) . ');background-size:cover;background-position:center;background-repeat:no-repeat;} ';
            }
        }
    }

    // タイトル/キャッチフレーズの表示制御（ロゴには影響させない）
    if ($show_title !== 1) {
        $css .= '.icpl-site-title{display:none !important;} ';
    }
    if ($show_tagline !== 1) {
        $css .= '.icpl-site-desc{display:none !important;} ';
    }

    wp_add_inline_style('icpl-ichimaru-plus-style', $css);
}, 20);

/**
 * Extract YouTube video ID from common URL forms.
 */
function icpl_extract_youtube_video_id($url) {
    $url = (string) $url;
    if ($url === '') {
        return '';
    }

    $video_id = '';

    // youtu.be/ID
    if (preg_match('~youtu\\.be/([A-Za-z0-9_-]{6,})~', $url, $m)) {
        $video_id = $m[1];
    }

    // youtube.com/watch?v=ID
    if ($video_id === '') {
        $q = wp_parse_url($url, PHP_URL_QUERY);
        if ($q) {
            parse_str($q, $args);
            if (!empty($args['v'])) {
                $video_id = preg_replace('~[^A-Za-z0-9_-]~', '', (string) $args['v']);
            }
        }
    }

    // youtube.com/embed/ID or youtube.com/shorts/ID
    if ($video_id === '' && preg_match('~youtube\\.com/(?:embed|shorts)/([A-Za-z0-9_-]{6,})~', $url, $m)) {
        $video_id = $m[1];
    }

    return (string) $video_id;
}

/**
 * Hero YouTube background: enqueue external JS from /assets/hero-youtube-bg.js
 * and pass the videoId via wp_localize_script.
 */
add_action('wp_enqueue_scripts', function () {
    // Only needed when selected
    $bg_type = (string) get_theme_mod('icpl_hero_bg_type', 'none');
    if ($bg_type !== 'youtube') {
        return;
    }

    // Front page hero usage only (avoid loading site-wide)
    if (!is_front_page()) {
        return;
    }

    $url = (string) get_theme_mod('icpl_hero_bg_youtube', '');
    if ($url === '') {
        return;
    }

    $start = (int) get_theme_mod('icpl_hero_bg_youtube_start', 0);
    if ($start < 0) { $start = 0; }

    $end = (int) get_theme_mod('icpl_hero_bg_youtube_end', 0);
    if ($end < 0) { $end = 0; }
    // end が start 以下の場合は「終了指定なし」とみなす
    if ($end > 0 && $end <= $start) {
        $end = 0;
    }

    $video_id = icpl_extract_youtube_video_id($url);
    if ($video_id === '') {
        return;
    }

    wp_enqueue_script(
        'icpl-hero-youtube-bg',
        get_template_directory_uri() . '/assets/hero-youtube-bg.js',
        array(),
        ICPL_THEME_VERSION,
        true
    );

    wp_localize_script(
        'icpl-hero-youtube-bg',
        'ICPL_HERO_YT',
        array(
            'videoId'  => $video_id,
            'start'    => $start,
            'end'      => $end,
            'selector' => '.icpl-hero',
        )
    );
}, 30);

/**
 * 色ユーティリティ：HEXを明るく/暗くする（hover用）
 */
function icpl_adjust_hex($hex, $steps) {
    $hex = ltrim((string)$hex, '#');
    if (strlen($hex) === 3) {
        $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
    }
    if (strlen($hex) !== 6) {
        return '#2563eb';
    }

    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));

    $r = max(0, min(255, $r + (int)$steps));
    $g = max(0, min(255, $g + (int)$steps));
    $b = max(0, min(255, $b + (int)$steps));

    return sprintf('#%02x%02x%02x', $r, $g, $b);
}

/**
 * Helpers
 */
function icpl_get_logo_or_title() {
    if (function_exists('the_custom_logo') && has_custom_logo()) {
        the_custom_logo();
    } else {
        echo '<h1 class="icpl-site-title"><a href="' . esc_url(home_url('/')) . '">' . esc_html(get_bloginfo('name')) . '</a></h1>';
    }
}

/**
 * SVGロゴ対応
 *
 * 推奨構成：
 *   /includes/
 *     icpl-svg.php
 *
 * functions.php は「入口」だけにして、機能は includes 配下に分離する。
 * （このテーマは単体で動く必要があるので、該当ファイルが無い場合はフォールバックで内蔵。）
 */
$icpl_svg_inc = get_template_directory() . '/includes/icpl-svg.php';
if (!file_exists($icpl_svg_inc)) {
    $icpl_svg_inc = get_template_directory() . '/inc/icpl-svg.php';
}

if (file_exists($icpl_svg_inc)) {
    require_once $icpl_svg_inc;
} else {
    /**
     * SVGロゴ対応（注意：SVGはスクリプト混入リスクがあるため、管理者のみに限定）
     * - これで「外観 → カスタマイズ → サイト基本情報 → ロゴ」でSVGを選べるようにする。
     * - 安全性を重視する場合は、SVGをPNGに変換するか、SVGサニタイズ対応プラグイン（例：Safe SVG）を推奨。
     */
    add_filter('upload_mimes', function ($mimes) {
        if (current_user_can('manage_options')) {
            $mimes['svg']  = 'image/svg+xml';
            $mimes['svgz'] = 'image/svg+xml';
        }
        return $mimes;
    });

    add_filter('wp_check_filetype_and_ext', function ($data, $file, $filename, $mimes) {
        if (!current_user_can('manage_options')) {
            return $data;
        }

        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        if ($ext === 'svg' || $ext === 'svgz') {
            $data['ext']  = $ext;
            $data['type'] = 'image/svg+xml';
        }
        return $data;
    }, 10, 4);

    // 管理画面のメディア一覧でSVGサムネが潰れるのを軽減
    add_action('admin_head', function () {
        echo '<style>.media-icon img[src$=".svg"], img[src$=".svg"], .attachment img[src$=".svg"]{width:100% !important;height:auto !important;}</style>';
    });
}