<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * バナー対象にする投稿タイプ一覧（内部リンク用）
 * - public かつ show_ui のもの
 * - attachment 等は除外
 */
function icpl_banner_widget_allowed_post_types() {
    $objs = get_post_types([
        'public'  => true,
        'show_ui' => true,
    ], 'objects');

    $allowed = [];
    if (is_array($objs)) {
        foreach ($objs as $slug => $obj) {
            if (!is_string($slug) || $slug === '') continue;
            if (in_array($slug, ['attachment', 'revision', 'nav_menu_item'], true)) continue;
            $allowed[] = $slug;
        }
    }

    // 念のため最低限
    if (!in_array('page', $allowed, true)) $allowed[] = 'page';
    if (!in_array('post', $allowed, true)) $allowed[] = 'post';

    return $allowed;
}

/**
 * ICPL Banner Widget (Page+ style)
 * - INTERNAL ONLY: select an internal post (page/post/custom post types) as target (no external URL)
 * - Uses target featured image (アイキャッチ) by default
 */

if (!class_exists('ICPL_Banner_Widget')):

class ICPL_Banner_Widget extends WP_Widget {

    public function __construct() {
        parent::__construct(
            'icpl_banner_widget',
            'ICPL バナー',
            [
                'classname'   => 'icpl_banner_widget',
                'description' => '固定ページ/投稿ページを選択してバナー（カード）を追加します。',
            ]
        );

        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_styles'], 20);
    }

    /**
     * 内部リンクとして許可する投稿タイプ一覧
     */
    private static function get_allowed_post_types() {
        return icpl_banner_widget_allowed_post_types();
    }

    private static function build_excerpt($post_id, $max = 120) {
        $post_id = (int) $post_id;
        if ($post_id <= 0) return '';

        $ex = has_excerpt($post_id) ? get_the_excerpt($post_id) : '';
        if (is_string($ex) && trim($ex) !== '') {
            return trim(wp_strip_all_tags($ex));
        }

        $content = get_post_field('post_content', $post_id);
        if (!is_string($content) || $content === '') return '';

        $plain = trim(wp_strip_all_tags($content));
        if ($plain === '') return '';

        $out = mb_substr($plain, 0, (int)$max);
        if (mb_strlen($plain) > (int)$max) $out .= '…';
        return $out;
    }

    public function widget($args, $instance) {
        $target_id = isset($instance['target_id']) ? (int)$instance['target_id'] : 0;

        $title = isset($instance['title']) ? trim((string)$instance['title']) : '';
        $desc  = isset($instance['desc']) ? trim((string)$instance['desc']) : '';
        $target_type = isset($instance['target_type']) ? (string)$instance['target_type'] : 'page';
        $external_url = isset($instance['external_url']) ? trim((string)$instance['external_url']) : '';
        $image_id = isset($instance['image_id']) ? (int)$instance['image_id'] : 0;
        $height = isset($instance['height']) ? (int)$instance['height'] : 152;
        $show_arrow = array_key_exists('show_arrow', $instance) ? (int)!empty($instance['show_arrow']) : 1;

        // 外部リンク
        if ($target_type === 'external') {
            $url = esc_url_raw($external_url);
            if (!is_string($url) || $url === '') return;

            // タイトル未入力ならURLを短く表示
            if ($title === '') {
                $title = preg_replace('~^https?://~', '', $url);
                $title = rtrim((string)$title, '/');
            }

            // 外部リンクでは本文からの自動生成はしない（未入力なら空のまま）
            // 背景画像は手動選択のみ
        } else {
            // 内部リンク
            if ($target_id <= 0) return;

            $p = get_post($target_id);
            if (!$p || $p->post_status !== 'publish') return;

            // 内部専用：public で UI に出る投稿タイプのみ（attachment 等は除外）
            $allowed_types = self::get_allowed_post_types();
            if (!in_array($p->post_type, $allowed_types, true)) return;

            if ($title === '') {
                $t = get_the_title($target_id);
                if (is_string($t)) $title = $t;
            }
            if ($desc === '') {
                $desc = self::build_excerpt($target_id, 120);
            }

            if (!$image_id && has_post_thumbnail($target_id)) {
                $image_id = (int) get_post_thumbnail_id($target_id);
            }

            $url = get_permalink($target_id);
            if (!is_string($url) || $url === '') return;
        }

        $bg_url = '';
        if ($image_id) {
            $tmp = wp_get_attachment_image_url($image_id, 'large');
            if (is_string($tmp)) $bg_url = $tmp;
        }

        if ($height < 110) $height = 110;
        if ($height > 360) $height = 360;

        echo $args['before_widget'];
        ?>
        <!-- ICPL Banner Widget: v2026-02-25 (internal) -->
        <div class="icpl-banner" style="--icpl-banner-minh: <?php echo (int)$height; ?>px;">
          <a class="icpl-banner__card" href="<?php echo esc_url($url); ?>">
            <span class="icpl-banner__card-bg" aria-hidden="true"<?php echo $bg_url ? ' style="background-image:url(' . esc_url($bg_url) . ');"' : ''; ?>></span>

            <span class="icpl-banner__body">
              <?php if ($title !== ''): ?>
                <span class="icpl-banner__title"><?php echo esc_html($title); ?></span>
              <?php endif; ?>

              <?php if ($desc !== ''): ?>
                <span class="icpl-banner__excerpt"><?php echo esc_html($desc); ?></span>
              <?php endif; ?>
            </span>

            <?php if ($show_arrow): ?>
              <span class="icpl-banner__arrow" aria-hidden="true"><span class="icpl-banner__arrow-ic">›</span></span>
            <?php endif; ?>
          </a>
        </div>
        <?php
        echo $args['after_widget'];
    }

    public function form($instance) {
        if (is_admin()) {
            wp_enqueue_media();
        }

        $allowed_types = self::get_allowed_post_types();
        $external_url = isset($instance['external_url']) ? (string)$instance['external_url'] : '';

        $target_type = isset($instance['target_type']) ? (string)$instance['target_type'] : 'page';
        if ($target_type !== 'external' && !in_array($target_type, $allowed_types, true)) {
            $target_type = in_array('page', $allowed_types, true) ? 'page' : (string)($allowed_types[0] ?? 'post');
        }

        $target_id = isset($instance['target_id']) ? (int)$instance['target_id'] : 0;

        $title = isset($instance['title']) ? (string)$instance['title'] : '';
        $desc  = isset($instance['desc']) ? (string)$instance['desc'] : '';
        $image_id = isset($instance['image_id']) ? (int)$instance['image_id'] : 0;
        $height = isset($instance['height']) ? (int)$instance['height'] : 152;
        $show_arrow = array_key_exists('show_arrow', $instance) ? (int)!empty($instance['show_arrow']) : 1;

        if ($height <= 0) $height = 152;

        $items = get_posts([
            'post_type'      => $target_type,
            'post_status'    => ['publish'],
            'posts_per_page' => 50,
            'orderby'        => 'date',
            'order'          => 'DESC',
        ]);

        // Nonce for AJAX
        $ajax_nonce = wp_create_nonce('icpl_banner_widget_fetch_posts');
        ?>
        <p style="margin:0 0 8px; opacity:.8;">
          内部の投稿（固定ページ/投稿/カスタム投稿）または外部リンクを選択できます。
        </p>

        <p>
          <label for="<?php echo esc_attr($this->get_field_id('target_type')); ?>">対象タイプ</label>
          <select class="widefat" id="<?php echo esc_attr($this->get_field_id('target_type')); ?>" name="<?php echo esc_attr($this->get_field_name('target_type')); ?>" data-icpl-banner-target="1" data-icpl-banner-nonce="<?php echo esc_attr($ajax_nonce); ?>">
            <?php
              $type_objs = get_post_types([
                  'public'  => true,
                  'show_ui' => true,
              ], 'objects');

              if (is_array($type_objs)) {
                  foreach ($allowed_types as $slug) {
                      if (!isset($type_objs[$slug])) continue;
                      $label = $type_objs[$slug]->labels->singular_name ?? $slug;
                      echo '<option value="' . esc_attr($slug) . '" ' . selected($target_type, $slug, false) . '>' . esc_html($label) . '</option>';
                  }
              }
              echo '<option value="external" ' . selected($target_type, 'external', false) . '>外部リンク</option>';
            ?>
          </select>
          <small style="opacity:.75;">※タイプ変更で一覧が更新されます（外部リンクの場合はURL入力欄が出ます）</small>
        </p>

        <p class="icpl-banner-external-url" style="display:<?php echo ($target_type === 'external') ? 'block' : 'none'; ?>;">
          <label for="<?php echo esc_attr($this->get_field_id('external_url')); ?>">外部URL</label>
          <input class="widefat" id="<?php echo esc_attr($this->get_field_id('external_url')); ?>" name="<?php echo esc_attr($this->get_field_name('external_url')); ?>" type="url" placeholder="https://example.com/" value="<?php echo esc_attr($external_url); ?>" />
          <small style="opacity:.75;">※外部リンクの場合、タイトル/説明は手入力（空ならURL表示）</small>
        </p>

        <p>
          <label for="<?php echo esc_attr($this->get_field_id('target_id')); ?>">リンク先（最新50件）</label>
          <select class="widefat" id="<?php echo esc_attr($this->get_field_id('target_id')); ?>" name="<?php echo esc_attr($this->get_field_name('target_id')); ?>" data-icpl-banner-post-select="1">
            <option value="0">— 選択 —</option>
            <?php if ($items): foreach ($items as $it): ?>
              <option value="<?php echo (int)$it->ID; ?>" <?php selected($target_id, (int)$it->ID); ?>>
                <?php echo esc_html(get_the_title($it->ID)); ?> (#<?php echo (int)$it->ID; ?>)
              </option>
            <?php endforeach; endif; ?>
          </select>
        </p>

        <hr style="border:0;border-top:1px solid rgba(127,127,127,.25);margin:12px 0;" />

        <p>
          <label for="<?php echo esc_attr($this->get_field_id('title')); ?>">タイトル（空なら対象ページのタイトル）</label>
          <input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>" name="<?php echo esc_attr($this->get_field_name('title')); ?>" type="text" value="<?php echo esc_attr($title); ?>" />
        </p>

        <p>
          <label for="<?php echo esc_attr($this->get_field_id('desc')); ?>">説明（空なら抜粋/本文から自動生成）</label>
          <textarea class="widefat" rows="3" id="<?php echo esc_attr($this->get_field_id('desc')); ?>" name="<?php echo esc_attr($this->get_field_name('desc')); ?>"><?php echo esc_textarea($desc); ?></textarea>
        </p>

        <p>
          <label for="<?php echo esc_attr($this->get_field_id('height')); ?>">高さ（px）</label>
          <input class="small-text" id="<?php echo esc_attr($this->get_field_id('height')); ?>" name="<?php echo esc_attr($this->get_field_name('height')); ?>" type="number" value="<?php echo (int)$height; ?>" min="110" max="360" />
        </p>

        <p>
          <label>
            <input type="checkbox" name="<?php echo esc_attr($this->get_field_name('show_arrow')); ?>" value="1" <?php checked($show_arrow, 1); ?> />
            右矢印を表示
          </label>
        </p>

        <p>
          <label>背景画像（任意：未指定なら対象ページのアイキャッチ）</label><br>
          <input type="hidden" id="<?php echo esc_attr($this->get_field_id('image_id')); ?>" name="<?php echo esc_attr($this->get_field_name('image_id')); ?>" value="<?php echo (int)$image_id; ?>" />
          <button class="button icpl-banner-media-select" type="button" data-target="#<?php echo esc_attr($this->get_field_id('image_id')); ?>">画像を選択</button>
          <button class="button icpl-banner-media-clear" type="button" data-target="#<?php echo esc_attr($this->get_field_id('image_id')); ?>">クリア</button>
          <?php if ($image_id): ?>
            <div class="icpl-banner-media-preview" style="margin-top:10px;">
              <?php echo wp_get_attachment_image($image_id, 'medium', false, ['style' => 'max-width:100%;height:auto;']); ?>
            </div>
          <?php endif; ?>
        </p>

        <script>
        (function(){
          if (typeof wp === 'undefined' || !wp.media) return;
          if (window.__icplBannerWidgetMediaBound) return;
          window.__icplBannerWidgetMediaBound = true;

          // 対象タイプ変更 → リンク先一覧をリアルタイム更新
          function findClosestWidgetRoot(el){
            // widgets.php / カスタマイザー / SiteOrigin で DOM が微妙に違うため、広めに拾う
            return el.closest('.widget-content') || el.closest('.so-widget-form') || el.closest('form') || document;
          }

          async function fetchPosts(postType, nonce){
            var ajax = window.ajaxurl || (window.icplAjax && window.icplAjax.ajaxurl) || '';
            if (!ajax) return null;

            var body = new URLSearchParams();
            body.append('action', 'icpl_banner_widget_fetch_posts');
            body.append('nonce', nonce);
            body.append('post_type', postType);

            var res = await fetch(ajax, {
              method: 'POST',
              credentials: 'same-origin',
              headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
              body: body.toString()
            });

            if (!res.ok) return null;
            var json = await res.json();
            if (!json || !json.success) return null;
            return (json.data && json.data.items) ? json.data.items : [];
          }

          function setOptions(selectEl, items, currentId){
            if (!selectEl) return;
            var cur = String(currentId || selectEl.value || '0');
            selectEl.innerHTML = '';

            var opt0 = document.createElement('option');
            opt0.value = '0';
            opt0.textContent = '— 選択 —';
            selectEl.appendChild(opt0);

            if (!items || !items.length) {
              var optEmpty = document.createElement('option');
              optEmpty.value = '0';
              optEmpty.textContent = '（公開済みがありません）';
              selectEl.appendChild(optEmpty);
              selectEl.value = '0';
              return;
            }

            items.forEach(function(it){
              var o = document.createElement('option');
              o.value = String(it.id);
              o.textContent = String(it.title || '') + ' (#' + String(it.id) + ')';
              selectEl.appendChild(o);
            });

            // 可能なら選択を維持
            var exists = Array.prototype.some.call(selectEl.options, function(o){ return o.value === cur; });
            selectEl.value = exists ? cur : '0';
          }

          document.addEventListener('change', async function(e){
            var sel = e.target;
            if (!sel || sel.getAttribute('data-icpl-banner-target') !== '1') return;

            var nonce = sel.getAttribute('data-icpl-banner-nonce') || '';
            var postType = sel.value;

            var root = findClosestWidgetRoot(sel);
            var postSelect = root.querySelector('select[data-icpl-banner-post-select="1"]');
            if (!postSelect) return;

            // 外部リンクUIの表示切替
            var extWrap = root.querySelector('.icpl-banner-external-url');
            if (extWrap) {
              extWrap.style.display = (postType === 'external') ? 'block' : 'none';
            }

            // 外部リンクの場合：リンク先一覧は無効化して終了
            if (postType === 'external') {
              postSelect.innerHTML = '';
              var optExt = document.createElement('option');
              optExt.value = '0';
              optExt.textContent = '（外部リンク）';
              postSelect.appendChild(optExt);
              postSelect.value = '0';
              return;
            }

            // ローディング表示
            var prevVal = postSelect.value;
            postSelect.innerHTML = '';
            var optLoading = document.createElement('option');
            optLoading.value = '0';
            optLoading.textContent = '読み込み中…';
            postSelect.appendChild(optLoading);
            postSelect.value = '0';

            var items = null;
            try {
              items = await fetchPosts(postType, nonce);
            } catch (err) {
              items = null;
            }

            if (items === null) {
              postSelect.innerHTML = '';
              var optErr = document.createElement('option');
              optErr.value = '0';
              optErr.textContent = '取得に失敗しました';
              postSelect.appendChild(optErr);
              postSelect.value = '0';
              return;
            }

            setOptions(postSelect, items, prevVal);
          });

          document.addEventListener('click', function(e){
            var btn = e.target.closest('.icpl-banner-media-select');
            if (!btn) return;
            e.preventDefault();

            var root = findClosestWidgetRoot(btn);
            var input = root.querySelector(btn.getAttribute('data-target'));
            if (!input) return;

            var frame = wp.media({ title: '画像を選択', button: { text: 'この画像を使用' }, multiple: false });
            frame.on('select', function(){
              var att = frame.state().get('selection').first().toJSON();
              if (att && att.id) {
                input.value = att.id;
                input.dispatchEvent(new Event('change', { bubbles: true }));
              }
            });
            frame.open();
          });

          document.addEventListener('click', function(e){
            var btn = e.target.closest('.icpl-banner-media-clear');
            if (!btn) return;
            e.preventDefault();

            var root = findClosestWidgetRoot(btn);
            var input = root.querySelector(btn.getAttribute('data-target'));
            if (!input) return;
            input.value = '';
            input.dispatchEvent(new Event('change', { bubbles: true }));

            // プレビューが出ている場合は消す（保存前でも視覚的に分かるように）
            var prev = root.querySelector('.icpl-banner-media-preview');
            if (prev) prev.remove();
          });
        })();
        </script>
        <?php
    }

    public function update($new_instance, $old_instance) {
        $instance = [];

        $allowed_types = self::get_allowed_post_types();
        $target_type = isset($new_instance['target_type']) ? (string)$new_instance['target_type'] : 'page';
        if ($target_type !== 'external' && !in_array($target_type, $allowed_types, true)) {
            $target_type = in_array('page', $allowed_types, true) ? 'page' : (string)($allowed_types[0] ?? 'post');
        }
        $instance['target_type'] = $target_type;

        $instance['target_id'] = isset($new_instance['target_id']) ? (int)$new_instance['target_id'] : 0;
        $instance['external_url'] = isset($new_instance['external_url']) ? esc_url_raw((string)$new_instance['external_url']) : '';

        $instance['title'] = isset($new_instance['title']) ? sanitize_text_field((string)$new_instance['title']) : '';
        $instance['desc']  = isset($new_instance['desc']) ? sanitize_textarea_field((string)$new_instance['desc']) : '';
        $instance['image_id'] = isset($new_instance['image_id']) ? (int)$new_instance['image_id'] : 0;
        $instance['height'] = isset($new_instance['height']) ? (int)$new_instance['height'] : 152;
        $instance['show_arrow'] = !empty($new_instance['show_arrow']) ? 1 : 0;

        if ($instance['height'] < 110) $instance['height'] = 110;
        if ($instance['height'] > 360) $instance['height'] = 360;

        return $instance;
    }

    public static function enqueue_styles() {
        $handle = 'icpl-theme-style';

        $css  = "/* ICPL Banner (Page+ style) */\n";
        $css .= ".icpl-banner{margin:18px 0 0;padding:0;background:transparent;border:0;}\n";
        $css .= ".icpl-banner .icpl-banner__card{box-sizing:border-box;cursor:pointer;display:flex;align-items:center;gap:16px;min-height:var(--icpl-banner-minh,152px);padding:18px 20px;position:relative;width:100%;text-decoration:none;color:inherit;overflow:hidden;border-radius:0;background:transparent;border:0;box-shadow:0 14px 34px rgba(0,0,0,.26);}\n";
        $css .= ".icpl-banner .icpl-banner__card:hover{transform:translateY(-2px);transition:transform .14s ease;box-shadow:0 24px 56px rgba(0,0,0,.28);}\n";
        $css .= ".icpl-banner .icpl-banner__card:focus-visible{outline:none;box-shadow:0 0 0 3px rgba(127,127,127,.22), 0 24px 56px rgba(0,0,0,.28);}\n";
        $css .= ".icpl-banner__card-bg{position:absolute;inset:0;pointer-events:none;z-index:0;background-color: rgba(255,255,255,.04);background-size:cover;background-position:center;filter:saturate(1.05) contrast(1.05);}\n";
        $css .= ".icpl-banner__card-bg::before{content:\"\";position:absolute;inset:0;opacity:.92;background: linear-gradient(90deg, rgba(0,0,0,.72), rgba(0,0,0,.45) 55%, rgba(0,0,0,.22));}\n";
        $css .= ".icpl-banner__card-bg::after{content:\"\";position:absolute;inset:0;opacity:.32;mix-blend-mode:overlay;background: radial-gradient(circle at 2px 2px, rgba(255,255,255,.28) 0 .6px, rgba(255,255,255,0) 1px) 0 0/4px 4px;}\n";
        $css .= ".icpl-banner .icpl-banner__card::after{content:\"\";position:absolute;inset:10px;border:1px solid rgba(255,255,255,.26);pointer-events:none;z-index:2;}\n";
        $css .= ".icpl-banner__body,.icpl-banner__arrow{position:relative;z-index:3;}\n";
        $css .= ".icpl-banner__body{display:flex;flex-direction:column;min-width:0;gap:6px;flex:1 1 auto;}\n";
        $css .= ".icpl-banner__title{font-weight:900;letter-spacing:.01em;line-height:1.2;font-size:1.06em;display:-webkit-box;-webkit-line-clamp:2;-webkit-box-orient:vertical;text-shadow:0 1px 0 rgba(0,0,0,.25);}\n";
        $css .= ".icpl-banner__excerpt{opacity:.86;font-size:.92em;line-height:1.45;white-space:pre-wrap;word-break:break-word;text-shadow:0 1px 0 rgba(0,0,0,.22);}\n";
        $css .= ".icpl-banner__arrow{flex:0 0 auto;display:flex;align-items:center;justify-content:center;}\n";
        $css .= ".icpl-banner__arrow-ic{font-weight:600;line-height:1;font-size:1.6em;opacity:.85;transition:transform .18s ease, opacity .18s ease;}\n";
        $css .= ".icpl-banner .icpl-banner__card:hover .icpl-banner__arrow-ic{transform:translateX(4px);opacity:1;}\n";
        $css .= "@media (max-width: 768px){.icpl-banner .icpl-banner__card{min-height:160px;padding:14px;gap:12px;}}\n";

        if (wp_style_is($handle, 'enqueued') || wp_style_is($handle, 'registered')) {
            wp_add_inline_style($handle, $css);
        } else {
            add_action('wp_head', function() use ($css){
                echo "\n<style id=\"icpl-banner-widget\">\n" . $css . "</style>\n";
            }, 99);
        }
    }
}

endif;

// 管理画面でメディアを使えるようにする（SiteOrigin含む）
add_action('admin_enqueue_scripts', function($hook){
    if (in_array($hook, ['widgets.php', 'customize.php', 'post.php', 'post-new.php'], true)) {
        wp_enqueue_media();
    }
});

// ウィジェット登録
add_action('widgets_init', function(){
    if (class_exists('ICPL_Banner_Widget')) {
        register_widget('ICPL_Banner_Widget');
    }
});

/**
 * Admin AJAX: 投稿一覧を取得（対象タイプ変更時にリアルタイム反映）
 */
add_action('wp_ajax_icpl_banner_widget_fetch_posts', function(){
    if (!current_user_can('edit_posts')) {
        wp_send_json_error(['message' => 'permission denied'], 403);
    }

    $nonce = isset($_POST['nonce']) ? (string) $_POST['nonce'] : '';
    if (!wp_verify_nonce($nonce, 'icpl_banner_widget_fetch_posts')) {
        wp_send_json_error(['message' => 'bad nonce'], 400);
    }

    $post_type = isset($_POST['post_type']) ? sanitize_key((string) $_POST['post_type']) : 'page';
    if ($post_type === 'external') {
        wp_send_json_error(['message' => 'external is not a post_type'], 400);
    }
    $allowed = icpl_banner_widget_allowed_post_types();
    if (!in_array($post_type, $allowed, true)) {
        wp_send_json_error(['message' => 'invalid post_type'], 400);
    }

    $items = get_posts([
        'post_type'      => $post_type,
        'post_status'    => ['publish'],
        'posts_per_page' => 50,
        'orderby'        => 'date',
        'order'          => 'DESC',
        'fields'         => 'ids',
    ]);

    $out = [];
    if ($items) {
        foreach ($items as $id) {
            $out[] = [
                'id'    => (int) $id,
                'title' => (string) get_the_title($id),
            ];
        }
    }

    wp_send_json_success(['items' => $out]);
});