<?php
/**
 * ICPL SVG Support
 *
 * - WordPress の「サイトロゴ」で SVG を扱えるようにする。
 * - SVG はスクリプト混入リスクがあるため、管理者（manage_options）のみに限定。
 * - 追加で簡易チェック（<script / on*属性 / javascript:）を検知したらアップロードを拒否。
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * 管理者のみ SVG/SVGZ をアップロード許可
 */
add_filter('upload_mimes', function ($mimes) {
    if (current_user_can('manage_options')) {
        $mimes['svg']  = 'image/svg+xml';
        $mimes['svgz'] = 'image/svg+xml';
    }
    return $mimes;
});

/**
 * WP が拡張子/ MIME を弾くケースの補正（管理者のみ）
 */
add_filter('wp_check_filetype_and_ext', function ($data, $file, $filename, $mimes) {
    if (!current_user_can('manage_options')) {
        return $data;
    }

    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    if ($ext === 'svg' || $ext === 'svgz') {
        $data['ext']  = $ext;
        $data['type'] = 'image/svg+xml';
    }
    return $data;
}, 10, 4);

/**
 * 危険パターンの簡易検知（管理者のみ）
 *  - 完全なサニタイズではない（厳密運用なら Safe SVG 等を推奨）
 */
add_filter('wp_handle_upload_prefilter', function ($file) {
    if (!current_user_can('manage_options')) {
        return $file;
    }

    $name = isset($file['name']) ? (string) $file['name'] : '';
    $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    if ($ext !== 'svg' && $ext !== 'svgz') {
        return $file;
    }

    $tmp = isset($file['tmp_name']) ? (string) $file['tmp_name'] : '';
    if ($tmp === '' || !is_readable($tmp)) {
        return $file;
    }

    // SVGZ は gzip のため中身チェックをスキップ（ここでは安全側に倒さず許可）。
    // 厳格にするなら SVGZ を禁止するのが確実。
    if ($ext === 'svgz') {
        return $file;
    }

    $content = file_get_contents($tmp);
    if ($content === false) {
        return $file;
    }

    $lc = strtolower($content);

    // ざっくり危険なものを検知
    $danger_patterns = array(
        '<script',
        'javascript:',
        'onload=',
        'onerror=',
        'onclick=',
        'onmouseover=',
        'onmouseenter=',
        'onmouseleave=',
        'onfocus=',
        'onblur=',
        '<iframe',
        '<object',
        '<embed',
    );

    foreach ($danger_patterns as $p) {
        if (strpos($lc, $p) !== false) {
            $file['error'] = __('SVG に危険な要素（script / on* 属性など）が含まれている可能性があるためアップロードできません。安全なSVG（サニタイズ済）を使用してください。', 'icpl-ichimaru-plus');
            return $file;
        }
    }

    return $file;
});

/**
 * 管理画面のメディア一覧で SVG サムネが潰れるのを軽減
 */
add_action('admin_head', function () {
    echo '<style>.media-icon img[src$=".svg"], img[src$=".svg"], .attachment img[src$=".svg"]{width:100% !important;height:auto !important;}</style>';
});
