<?php
/**
 * Latest News feature (ICPL)
 *
 * - Customizer settings:
 *   - Enable / disable
 *   - Section title (default: 新着情報)
 *   - Target post types (comma separated)
 *   - Include pages or not
 *   - Number of posts
 *
 * Usage:
 *   Call icpl_render_latest_news(); from front-page.php or a template-part.
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get available public post types for Latest News selector.
 * - Excludes attachments and internal types.
 * - Returns: slug => label
 */
function icpl_latest_news_get_available_post_types() {
    $pts = get_post_types(array('public' => true), 'objects');

    $out = array();
    foreach ($pts as $slug => $obj) {
        if (in_array($slug, array('attachment', 'nav_menu_item', 'revision'), true)) {
            continue;
        }
        // pages are controlled by a separate toggle
        if ($slug === 'page') {
            continue;
        }

        $label = !empty($obj->labels->singular_name) ? $obj->labels->singular_name : $slug;
        $out[$slug] = $label;
    }

    // Ensure "post" exists as fallback
    if (!isset($out['post'])) {
        $out = array('post' => '投稿') + $out;
    }

    return $out;
}

// --------------------------------------------------
// Debug marker: confirms this file is loaded
// (visible only to admins in page source and footer)
// --------------------------------------------------
add_action('wp_footer', function () {
    if (!current_user_can('manage_options')) {
        return;
    }
    $stamp = gmdate('Y-m-d H:i:s') . ' UTC';
    echo "\n<!-- ICPL_LATEST_NEWS_LOADED {$stamp} -->\n";
}, 9999);

// --------------------------------------------------
// Shortcode for quick testing: [icpl_latest_news]
// --------------------------------------------------
add_shortcode('icpl_latest_news', function () {
    ob_start();
    icpl_render_latest_news();
    return (string) ob_get_clean();
});

/**
 * --------------------------------------------------
 * Customizer settings
 * --------------------------------------------------
 */
add_action('customize_register', function ($wp_customize) {

    $wp_customize->add_section('icpl_latest_news', array(
        'title'       => __('新着情報', 'icpl-ichimaru-plus'),
        'priority'    => 40,
        'description' => __('フロントページに表示する新着情報の設定です。', 'icpl-ichimaru-plus'),
    ));

    // Enable / Disable
    $wp_customize->add_setting('icpl_latest_news_enable', array(
        'default'           => 1,
        'sanitize_callback' => function ($v) { return $v ? 1 : 0; },
    ));

    $wp_customize->add_control('icpl_latest_news_enable', array(
        'label'   => __('新着情報を表示する', 'icpl-ichimaru-plus'),
        'section' => 'icpl_latest_news',
        'type'    => 'checkbox',
    ));

    // Title
    $wp_customize->add_setting('icpl_latest_news_title', array(
        'default'           => __('新着情報', 'icpl-ichimaru-plus'),
        'sanitize_callback' => 'sanitize_text_field',
    ));

    $wp_customize->add_control('icpl_latest_news_title', array(
        'label'   => __('表示タイトル', 'icpl-ichimaru-plus'),
        'section' => 'icpl_latest_news',
        'type'    => 'text',
    ));

    // Target post types (one setting per post type: Customizer-native & stable)
    $available_pts = icpl_latest_news_get_available_post_types();

    foreach ($available_pts as $slug => $label) {
        $setting_id = 'icpl_latest_news_pt_' . $slug;

        $wp_customize->add_setting($setting_id, array(
            'default'           => ($slug === 'post') ? 1 : 0,
            'sanitize_callback' => function ($v) { return $v ? 1 : 0; },
        ));

        $wp_customize->add_control($setting_id, array(
            'label'   => sprintf(__('%s を含める', 'icpl-ichimaru-plus'), $label),
            'section' => 'icpl_latest_news',
            'type'    => 'checkbox',
        ));
    }

    // Include pages
    $wp_customize->add_setting('icpl_latest_news_include_pages', array(
        'default'           => 0,
        'sanitize_callback' => function ($v) { return $v ? 1 : 0; },
    ));

    $wp_customize->add_control('icpl_latest_news_include_pages', array(
        'label'   => __('固定ページを含める', 'icpl-ichimaru-plus'),
        'section' => 'icpl_latest_news',
        'type'    => 'checkbox',
    ));

    // Number of posts
    $wp_customize->add_setting('icpl_latest_news_count', array(
        'default'           => 5,
        'sanitize_callback' => 'absint',
    ));

    $wp_customize->add_control('icpl_latest_news_count', array(
        'label'       => __('表示件数', 'icpl-ichimaru-plus'),
        'section'     => 'icpl_latest_news',
        'type'        => 'number',
        'input_attrs' => array(
            'min'  => 1,
            'max'  => 20,
            'step' => 1,
        ),
    ));
});

// Customizer UI behavior: update the hidden comma-separated setting when checkboxes change.
// NOTE:
// Some WP builds/plugins rely on the *linked input element* change event (not only setting.set)
// to mark the changeset as dirty. So we update BOTH:
//  - setting.set(...)
//  - hidden input value + dispatch change/input events
// DELETED per instructions

/**
 * --------------------------------------------------
 * Data fetch helper
 * --------------------------------------------------
 */
function icpl_get_latest_news_posts() {

    $enable = (int) get_theme_mod('icpl_latest_news_enable', 1);
    if ($enable !== 1) {
        return array();
    }

    $available = icpl_latest_news_get_available_post_types();
    $types = array();

    foreach (array_keys($available) as $slug) {
        if ((int) get_theme_mod('icpl_latest_news_pt_' . $slug, ($slug === 'post') ? 1 : 0) === 1) {
            $types[] = $slug;
        }
    }

    if ((int) get_theme_mod('icpl_latest_news_include_pages', 0) === 1) {
        $types[] = 'page';
    }

    if (empty($types)) {
        return array();
    }

    $count = (int) get_theme_mod('icpl_latest_news_count', 5);
    if ($count < 1) {
        $count = 5;
    }

    $query = new WP_Query(array(
        'post_type'           => array_unique($types),
        'post_status'         => 'publish',
        'posts_per_page'      => $count,
        'orderby'             => 'date',
        'order'               => 'DESC',
        'ignore_sticky_posts' => true,
        'no_found_rows'       => true,
    ));

    return $query->have_posts() ? $query->posts : array();
}

/**
 * --------------------------------------------------
 * Render function (front side)
 * --------------------------------------------------
 */
function icpl_render_latest_news() {

    $enable = (int) get_theme_mod('icpl_latest_news_enable', 1);
    if ($enable !== 1) {
        return;
    }

    $posts = icpl_get_latest_news_posts();
    // Admin-only debug: shows why it might not render
    if (current_user_can('manage_options')) {
        $available = icpl_latest_news_get_available_post_types();
        $types = array();
        foreach (array_keys($available) as $slug) {
            if ((int) get_theme_mod('icpl_latest_news_pt_' . $slug, ($slug === 'post') ? 1 : 0) === 1) {
                $types[] = $slug;
            }
        }
        if ((int) get_theme_mod('icpl_latest_news_include_pages', 0) === 1) {
            $types[] = 'page';
        }
        $types_raw = implode(',', $types);
        $inc_page  = (int) get_theme_mod('icpl_latest_news_include_pages', 0);
        $count     = (int) get_theme_mod('icpl_latest_news_count', 5);
        echo "\n<!-- ICPL_LATEST_NEWS_RENDER enable=1 types={$types_raw} include_pages={$inc_page} count={$count} -->\n";
    }
    if (empty($posts)) {
        return;
    }

    $title = (string) get_theme_mod('icpl_latest_news_title', __('新着情報', 'icpl-ichimaru-plus'));
    ?>

    <section class="icpl-section icpl-latest-news">
        <div class="icpl-container">
            <?php if ($title !== ''): ?>
                <h2 class="icpl-section-title"><?php echo esc_html($title); ?></h2>
            <?php endif; ?>

            <ul class="icpl-latest-news-list">
                <?php foreach ($posts as $p): ?>
                    <?php
                        // Ensure template tags (get_the_date(), the_permalink(), the_title(), etc.)
                        // point to the current item.
                        $GLOBALS['post'] = $p;
                        setup_postdata($p);
                        $pid = isset($p->ID) ? (int) $p->ID : 0;
                    ?>
                    <li class="icpl-latest-news-item">
                        <time class="icpl-latest-news-date" datetime="<?php echo esc_attr(get_the_date('c', $pid)); ?>">
                            <?php echo esc_html(get_the_date('', $pid)); ?>
                        </time>
                        <a class="icpl-latest-news-link" href="<?php echo esc_url(get_permalink($pid)); ?>">
                            <?php echo esc_html(get_the_title($pid)); ?>
                        </a>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
    </section>

    <?php
    wp_reset_postdata();
}