<?php
/**
 * Page Hero Metaboxes
 *
 * Adds per-page settings for the Page Hero template part.
 * - Enable Collage Hero
 * - Select multiple images for collage
 * - Optional overlay (0-90)
 * - Overlay style ("solid" or "dots")
 *
 * どの投稿タイプに適用するかはカスタマイザーで選択可能です（初期状態は固定ページのみ）。
 *
 * Stored post meta:
 * - icpl_hero_collage_enable ("1" or "0")
 * - icpl_hero_collage_images (comma-separated attachment IDs)
 * - icpl_hero_collage_overlay (0-90)
 * - icpl_hero_overlay_style ("solid" or "dots")
 */

if (!defined('ABSPATH')) {
    exit;
}

class ICPL_Page_Hero_Metaboxes {

    const NONCE_ACTION = 'icpl_page_hero_metabox_save';
    const NONCE_NAME   = 'icpl_page_hero_metabox_nonce';

    public static function init() {
        // Customizer (適用先投稿タイプ)
        add_action('customize_register', [__CLASS__, 'register_customizer']);

        add_action('add_meta_boxes', [__CLASS__, 'add_metabox']);
        add_action('save_post', [__CLASS__, 'save_metabox'], 10, 2);

        // Media frame + UI behavior
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin_assets']);
        add_action('admin_footer-post.php', [__CLASS__, 'print_admin_js']);
        add_action('admin_footer-post-new.php', [__CLASS__, 'print_admin_js']);
    }

    /**
     * Customizer でONになっている投稿タイプ一覧を取得
     * - デフォルトは page のみ true
     *
     * @return string[]
     */
    private static function get_enabled_post_types(): array {
        $post_types = get_post_types(['public' => true], 'objects');
        $enabled = [];

        foreach ($post_types as $slug => $obj) {
            // 管理用/特殊なものは除外
            if (in_array($slug, ['attachment', 'nav_menu_item', 'revision', 'custom_css', 'customize_changeset'], true)) {
                continue;
            }

            $default = ($slug === 'page');
            $raw = get_theme_mod('icpl_page_hero_apply_' . $slug, $default ? 1 : 0);

            // Accept legacy boolean true, but otherwise require explicit "1"
            $is_on = ($raw === true || (string) $raw === '1');

            if ($is_on) {
                $enabled[] = $slug;
            }
        }

        // 念のため page は常に配列要素として扱えるように（ただしOFFにできる要件なら削除してもOK）
        // 今回は「デフォルトON」なので、OFFも許可しつつ、enabled が空なら page を入れる。
        if (empty($enabled)) {
            $enabled[] = 'page';
        }

        return array_values(array_unique($enabled));
    }

    /**
     * この画面（投稿タイプ）でメタボックスを出してよいか
     */
    private static function is_enabled_post_type(string $post_type): bool {
        return in_array($post_type, self::get_enabled_post_types(), true);
    }

    /**
     * 外観 → カスタマイズ に「ページヒーロー」適用先（投稿タイプ）を追加
     */
    public static function register_customizer($wp_customize) {
        if (!($wp_customize instanceof WP_Customize_Manager)) {
            return;
        }

        $section_id = 'icpl_section_page_hero';

        $wp_customize->add_section($section_id, [
            'title'       => 'ページヒーロー',
            'priority'    => 180,
            'description' => 'ヒーロー（コラージュ/暗幕など）のメタボックスを表示する投稿タイプを選択します。デフォルトは固定ページのみONです。',
        ]);

        $post_types = get_post_types(['public' => true], 'objects');

        foreach ($post_types as $slug => $obj) {
            if (in_array($slug, ['attachment', 'nav_menu_item', 'revision', 'custom_css', 'customize_changeset'], true)) {
                continue;
            }

            $setting_id = 'icpl_page_hero_apply_' . $slug;
            $default = ($slug === 'page');

            $wp_customize->add_setting($setting_id, [
                'default'           => $default,
                'type'              => 'theme_mod',
                'capability'        => 'edit_theme_options',
                'sanitize_callback' => function($v){
                    // Customizer checkbox can come as bool, int, or string.
                    // Store as 1/0 to avoid truthy strings causing unexpected ON state.
                    return ($v === true || $v === 1 || $v === '1' || $v === 'on') ? 1 : 0;
                },
            ]);

            $label = !empty($obj->labels->singular_name) ? $obj->labels->singular_name : $slug;
            $wp_customize->add_control($setting_id, [
                'section' => $section_id,
                'type'    => 'checkbox',
                'label'   => $label . '（' . $slug . '）',
            ]);
        }
    }

    public static function add_metabox() {
        $post_types = self::get_enabled_post_types();
        foreach ($post_types as $pt) {
            add_meta_box(
                'icpl_page_hero_metabox',
                'ページヒーロー',
                [__CLASS__, 'render_metabox'],
                $pt,
                'normal',
                'high'
            );
        }
    }

    public static function enqueue_admin_assets($hook) {
        // Only for enabled post type edit screens
        if ($hook !== 'post.php' && $hook !== 'post-new.php') {
            return;
        }

        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || !is_string($screen->post_type) || !self::is_enabled_post_type($screen->post_type)) {
            return;
        }

        // Required for wp.media
        wp_enqueue_media();
    }

    public static function render_metabox($post) {
        wp_nonce_field(self::NONCE_ACTION, self::NONCE_NAME);

        $enable = get_post_meta($post->ID, 'icpl_hero_collage_enable', true);
        $enable = ((string)$enable === '1') ? '1' : '0';

        $ids_raw = get_post_meta($post->ID, 'icpl_hero_collage_images', true);
        $ids_raw = is_string($ids_raw) ? trim($ids_raw) : '';

        $overlay = get_post_meta($post->ID, 'icpl_hero_collage_overlay', true);
        $overlay = is_numeric($overlay) ? (int)$overlay : 20;
        $overlay = max(0, min(90, $overlay));

        $overlay_style = get_post_meta($post->ID, 'icpl_hero_overlay_style', true);
        $overlay_style = ($overlay_style === 'dots') ? 'dots' : 'solid';

        // Parse IDs for preview
        $ids = [];
        if ($ids_raw !== '') {
            $parts = preg_split('/\s*,\s*/', $ids_raw, -1, PREG_SPLIT_NO_EMPTY);
            $ids = array_values(array_filter(array_map('intval', (array)$parts), function($v){ return $v > 0; }));
            $ids = array_values(array_unique($ids));
        }

        ?>
        <style>
            .icpl-ph-row { margin: 12px 0; }
            .icpl-ph-help { color: #666; margin-top: 6px; }
            .icpl-ph-thumbs { display:flex; flex-wrap:wrap; gap:8px; margin-top:10px; }
            .icpl-ph-thumb { width:72px; height:72px; border:1px solid #ddd; border-radius:6px; overflow:hidden; background:#fafafa; display:flex; align-items:center; justify-content:center; }
            .icpl-ph-thumb img { width:100%; height:100%; object-fit:cover; display:block; }
            .icpl-ph-actions { display:flex; gap:8px; align-items:center; }
            .icpl-ph-muted { opacity: .8; }
            .icpl-ph-hidden { display:none; }
        </style>

        <div class="icpl-page-hero-metabox" data-post-id="<?php echo esc_attr($post->ID); ?>">

            <div class="icpl-ph-row">
                <label>
                    <input type="checkbox" name="icpl_hero_collage_enable" value="1" <?php checked($enable, '1'); ?> />
                    ヒーローをコラージュにする
                </label>
                <div class="icpl-ph-help">ONにすると、固定ページのヒーローがアイキャッチではなく、下で選んだ複数画像のコラージュになります。</div>
            </div>

            <div class="icpl-ph-row icpl-ph-collage-fields <?php echo ($enable === '1') ? '' : 'icpl-ph-muted'; ?>">
                <div class="icpl-ph-actions">
                    <button type="button" class="button icpl-ph-select">画像を選択</button>
                    <button type="button" class="button icpl-ph-clear">クリア</button>
                    <span class="icpl-ph-help">複数選択できます。おすすめ：6〜12枚</span>
                </div>

                <input type="hidden" name="icpl_hero_collage_images" class="icpl-ph-ids" value="<?php echo esc_attr($ids_raw); ?>" />

                <div class="icpl-ph-thumbs">
                    <?php
                    if (!empty($ids)) {
                        foreach ($ids as $attachment_id) {
                            $src = wp_get_attachment_image_url($attachment_id, 'thumbnail');
                            if (!$src) {
                                continue;
                            }
                            echo '<div class="icpl-ph-thumb" data-id="' . esc_attr($attachment_id) . '"><img src="' . esc_url($src) . '" alt="" /></div>';
                        }
                    } else {
                        echo '<div class="icpl-ph-help">まだ画像が選択されていません。</div>';
                    }
                    ?>
                </div>

                <div class="icpl-ph-row" style="margin-top:14px;">
                    <label for="icpl_hero_collage_overlay">暗幕（オーバーレイ）</label>
                    <div style="display:flex; align-items:center; gap:10px; margin-top:6px;">
                        <input id="icpl_hero_collage_overlay" type="range" min="0" max="90" step="1" name="icpl_hero_collage_overlay" value="<?php echo esc_attr($overlay); ?>" />
                        <span class="icpl-ph-overlay-val"><?php echo esc_html($overlay); ?>%</span>
                    </div>
                    <div class="icpl-ph-help">文字が読みにくい場合は 15〜35% あたりが使いやすいです。</div>
                </div>

                <div class="icpl-ph-row" style="margin-top:10px;">
                    <label for="icpl_hero_overlay_style">暗幕スタイル</label>
                    <div style="margin-top:6px;">
                        <select id="icpl_hero_overlay_style" name="icpl_hero_overlay_style">
                            <option value="solid" <?php selected($overlay_style, 'solid'); ?>>ベタ（黒）</option>
                            <option value="dots" <?php selected($overlay_style, 'dots'); ?>>ハーフトーン（ドット）</option>
                        </select>
                    </div>
                    <div class="icpl-ph-help">ページごとに暗幕を「ベタ」か「ドット」から選択できます。</div>
                </div>
            </div>

            <div class="icpl-ph-row icpl-ph-help">
                仕組み：この設定は <code>template-parts/page-hero.php</code> が参照します。
            </div>

        </div>
        <?php
    }

    public static function save_metabox($post_id, $post) {
        // Nonce
        if (!isset($_POST[self::NONCE_NAME]) || !wp_verify_nonce($_POST[self::NONCE_NAME], self::NONCE_ACTION)) {
            return;
        }

        // Autosave / revisions
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }

        // 対象投稿タイプのみ
        if (!($post instanceof WP_Post) || !is_string($post->post_type) || !self::is_enabled_post_type($post->post_type)) {
            return;
        }

        // Capability
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Enable
        $enable = isset($_POST['icpl_hero_collage_enable']) ? '1' : '0';
        update_post_meta($post_id, 'icpl_hero_collage_enable', $enable);

        // Images (comma-separated IDs)
        $ids_raw = isset($_POST['icpl_hero_collage_images']) ? (string)$_POST['icpl_hero_collage_images'] : '';
        $ids_raw = trim($ids_raw);

        if ($ids_raw === '') {
            delete_post_meta($post_id, 'icpl_hero_collage_images');
        } else {
            // Sanitize to numeric IDs
            $parts = preg_split('/\s*,\s*/', $ids_raw, -1, PREG_SPLIT_NO_EMPTY);
            $ids = array_values(array_filter(array_map('intval', (array)$parts), function($v){ return $v > 0; }));
            $ids = array_values(array_unique($ids));

            update_post_meta($post_id, 'icpl_hero_collage_images', implode(',', $ids));
        }

        // Overlay
        $overlay = isset($_POST['icpl_hero_collage_overlay']) ? (int)$_POST['icpl_hero_collage_overlay'] : 20;
        $overlay = max(0, min(90, $overlay));
        update_post_meta($post_id, 'icpl_hero_collage_overlay', (string)$overlay);

        // Overlay style
        $overlay_style = isset($_POST['icpl_hero_overlay_style']) ? $_POST['icpl_hero_overlay_style'] : 'solid';
        $overlay_style = ($overlay_style === 'dots') ? 'dots' : 'solid';
        update_post_meta($post_id, 'icpl_hero_overlay_style', $overlay_style);
    }

    public static function print_admin_js() {
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || !is_string($screen->post_type) || !self::is_enabled_post_type($screen->post_type)) {
            return;
        }

        ?>
        <script>
        (function(){
            'use strict';

            function qs(root, sel){ return root.querySelector(sel); }
            function qsa(root, sel){ return Array.prototype.slice.call(root.querySelectorAll(sel)); }

            var box = document.querySelector('.icpl-page-hero-metabox');
            if (!box) return;

            var enable = qs(box, 'input[name="icpl_hero_collage_enable"]');
            var idsInput = qs(box, '.icpl-ph-ids');
            var thumbs = qs(box, '.icpl-ph-thumbs');
            var btnSelect = qs(box, '.icpl-ph-select');
            var btnClear = qs(box, '.icpl-ph-clear');
            var overlayRange = qs(box, 'input[name="icpl_hero_collage_overlay"]');
            var overlayVal = qs(box, '.icpl-ph-overlay-val');
            var fields = qs(box, '.icpl-ph-collage-fields');

            function setMuted(isMuted){
                if (!fields) return;
                if (isMuted) {
                    fields.classList.add('icpl-ph-muted');
                } else {
                    fields.classList.remove('icpl-ph-muted');
                }
            }

            function renderThumbs(ids){
                if (!thumbs) return;
                thumbs.innerHTML = '';

                if (!ids || !ids.length) {
                    var help = document.createElement('div');
                    help.className = 'icpl-ph-help';
                    help.textContent = 'まだ画像が選択されていません。';
                    thumbs.appendChild(help);
                    return;
                }

                ids.forEach(function(id){
                    var el = document.createElement('div');
                    el.className = 'icpl-ph-thumb';
                    el.setAttribute('data-id', id);
                    el.textContent = '...';
                    thumbs.appendChild(el);

                    // Try to fetch thumb URL from wp.media, fallback to empty tile
                    if (window.wp && wp.media && wp.media.attachment) {
                        var att = wp.media.attachment(id);
                        att.fetch().then(function(){
                            var json = att.toJSON();
                            var url = (json && json.sizes && json.sizes.thumbnail && json.sizes.thumbnail.url) ? json.sizes.thumbnail.url : (json && json.url ? json.url : '');
                            if (url) {
                                el.textContent = '';
                                var img = document.createElement('img');
                                img.src = url;
                                img.alt = '';
                                el.appendChild(img);
                            } else {
                                el.textContent = '';
                            }
                        }).catch(function(){
                            el.textContent = '';
                        });
                    } else {
                        el.textContent = '';
                    }
                });
            }

            function parseIds(value){
                if (!value) return [];
                return value.split(',').map(function(s){ return parseInt(String(s).trim(), 10); }).filter(function(n){ return n > 0; });
            }

            function uniq(ids){
                var seen = {};
                var out = [];
                ids.forEach(function(id){
                    if (!seen[id]) { seen[id] = true; out.push(id); }
                });
                return out;
            }

            function syncOverlayLabel(){
                if (!overlayRange || !overlayVal) return;
                overlayVal.textContent = overlayRange.value + '%';
            }

            // Toggle UI
            if (enable) {
                enable.addEventListener('change', function(){
                    setMuted(!enable.checked);
                });
                setMuted(!enable.checked);
            }

            // Overlay label
            if (overlayRange) {
                overlayRange.addEventListener('input', syncOverlayLabel);
                syncOverlayLabel();
            }

            // Media frame
            var frame = null;
            if (btnSelect) {
                btnSelect.addEventListener('click', function(e){
                    e.preventDefault();

                    if (!window.wp || !wp.media) {
                        alert('メディア機能を読み込めませんでした。');
                        return;
                    }

                    if (frame) {
                        frame.open();
                        return;
                    }

                    frame = wp.media({
                        title: 'コラージュ画像を選択',
                        library: { type: 'image' },
                        multiple: true,
                        button: { text: 'この画像を使う' }
                    });

                    frame.on('select', function(){
                        var selection = frame.state().get('selection');
                        var ids = [];
                        selection.each(function(att){
                            ids.push(att.get('id'));
                        });
                        ids = uniq(ids);
                        idsInput.value = ids.join(',');
                        renderThumbs(ids);
                    });

                    frame.open();
                });
            }

            if (btnClear) {
                btnClear.addEventListener('click', function(e){
                    e.preventDefault();
                    idsInput.value = '';
                    renderThumbs([]);
                });
            }

            // Initial thumbs (keep existing HTML if already rendered; otherwise render from ids)
            // If thumbnails are currently the help text, update from IDs.
            var current = parseIds(idsInput ? idsInput.value : '');
            if (current && current.length) {
                // If current thumbs contains only help text, rerender.
                var hasTile = thumbs && thumbs.querySelector('.icpl-ph-thumb');
                if (!hasTile) {
                    renderThumbs(current);
                }
            }

        })();
        </script>
        <?php
    }
}

ICPL_Page_Hero_Metaboxes::init();
