<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * ICPL Typography: Japanese font switcher
 *
 * - Customizer control: icpl_font_family (in section: icpl_colors)
 * - Enqueue selected Google Font only when needed
 * - Output CSS var --icpl-font and apply to body + form controls
 * - Apply to Block Editor
 */

/**
 * Customizer: add font selector.
 *
 * Note:
 * - `icpl_colors` section is registered elsewhere and may be added after this file is included.
 * - If we require the section to exist too early, the control never appears.
 * - So we run at a later priority and fall back to `title_tagline` when needed.
 */
add_action('customize_register', function ($wp_customize) {

    if (!is_object($wp_customize)) {
        return;
    }

    // Prefer the theme's section, but fall back safely.
    $section_id = $wp_customize->get_section('icpl_colors') ? 'icpl_colors' : 'title_tagline';

    $wp_customize->add_setting('icpl_font_family', array(
        'default'           => 'system_sans',
        'sanitize_callback' => function ($v) {
            $allowed = array(
                'system_sans',
                'system_serif',
                'noto_sans_jp',
                'noto_serif_jp',
                'zen_kaku_gothic',
                'zen_old_mincho',
                'biz_udpgothic',
                'biz_udmincho',
            );
            return in_array($v, $allowed, true) ? $v : 'system_sans';
        },
        'transport'         => 'refresh',
    ));

    $wp_customize->add_control('icpl_font_family', array(
        'label'       => __('フォント（日本語）', 'icpl-ichimaru-plus'),
        'description' => __('表示フォントを選択します。速度優先なら「システム」を推奨。Google Fontsは必要時のみ読み込みます。', 'icpl-ichimaru-plus'),
        'section'     => $section_id,
        'type'        => 'select',
        'choices'     => array(
            'system_sans'     => __('システム：ゴシック（最速）', 'icpl-ichimaru-plus'),
            'system_serif'    => __('システム：明朝（最速）', 'icpl-ichimaru-plus'),
            'noto_sans_jp'    => __('Noto Sans JP（Google）', 'icpl-ichimaru-plus'),
            'noto_serif_jp'   => __('Noto Serif JP（Google）', 'icpl-ichimaru-plus'),
            'zen_kaku_gothic' => __('Zen Kaku Gothic New（Google）', 'icpl-ichimaru-plus'),
            'zen_old_mincho'  => __('Zen Old Mincho（Google）', 'icpl-ichimaru-plus'),
            'biz_udpgothic'   => __('BIZ UDPGothic（Google）', 'icpl-ichimaru-plus'),
            'biz_udmincho'    => __('BIZ UDMincho（Google）', 'icpl-ichimaru-plus'),
        ),
        'priority'    => 40,
    ));

}, 50);

/**
 * Shared: font stacks and google font query.
 */
function icpl_typography_get_font_stacks() {
    return array(
        // System (fast)
        'system_sans'  => 'system-ui, -apple-system, BlinkMacSystemFont, "Hiragino Sans", "Hiragino Kaku Gothic ProN", "Yu Gothic", "YuGothic", Meiryo, "Noto Sans JP", sans-serif',
        'system_serif' => '"Hiragino Mincho ProN", "Yu Mincho", "YuMincho", "MS PMincho", "Noto Serif JP", serif',

        // Google
        'noto_sans_jp'    => '"Noto Sans JP", "Hiragino Sans", "Yu Gothic", Meiryo, sans-serif',
        'noto_serif_jp'   => '"Noto Serif JP", "Hiragino Mincho ProN", "Yu Mincho", serif',
        'zen_kaku_gothic' => '"Zen Kaku Gothic New", "Noto Sans JP", "Hiragino Sans", "Yu Gothic", Meiryo, sans-serif',
        'zen_old_mincho'  => '"Zen Old Mincho", "Noto Serif JP", "Hiragino Mincho ProN", "Yu Mincho", serif',
        'biz_udpgothic'   => '"BIZ UDPGothic", "Noto Sans JP", "Hiragino Sans", "Yu Gothic", Meiryo, sans-serif',
        'biz_udmincho'    => '"BIZ UDMincho", "Noto Serif JP", "Hiragino Mincho ProN", "Yu Mincho", serif',
    );
}

function icpl_typography_get_google_query_map() {
    return array(
        'noto_sans_jp'    => 'family=Noto+Sans+JP:wght@400;700&display=swap',
        'noto_serif_jp'   => 'family=Noto+Serif+JP:wght@400;700&display=swap',
        'zen_kaku_gothic' => 'family=Zen+Kaku+Gothic+New:wght@400;700&display=swap',
        'zen_old_mincho'  => 'family=Zen+Old+Mincho:wght@400;700&display=swap',
        'biz_udpgothic'   => 'family=BIZ+UDPGothic:wght@400;700&display=swap',
        'biz_udmincho'    => 'family=BIZ+UDMincho:wght@400;700&display=swap',
    );
}

/**
 * Enqueue selected Google Font only when needed.
 * Priority 19 so it loads before the theme inline CSS (priority 20) and our font inline CSS (priority 21).
 */
add_action('wp_enqueue_scripts', function () {

    $font_key = (string) get_theme_mod('icpl_font_family', 'system_sans');

    $map = icpl_typography_get_google_query_map();
    if (!isset($map[$font_key])) {
        return;
    }

    $href = 'https://fonts.googleapis.com/css2?' . $map[$font_key];

    $ver = defined('ICPL_THEME_VERSION') ? ICPL_THEME_VERSION : null;

    wp_enqueue_style(
        'icpl-font-ja',
        $href,
        array(),
        $ver
    );
}, 19);

/**
 * Output CSS variable and apply.
 * Priority 21 to come after the theme's inline variables.
 */
add_action('wp_enqueue_scripts', function () {

    $font_key = (string) get_theme_mod('icpl_font_family', 'system_sans');

    $stacks = icpl_typography_get_font_stacks();
    $font_family = isset($stacks[$font_key]) ? $stacks[$font_key] : $stacks['system_sans'];

    $css  = ':root{--icpl-font:' . $font_family . ';}';
    $css .= 'body{font-family:var(--icpl-font);}';
    $css .= 'button,input,select,textarea{font-family:inherit;}';

    wp_add_inline_style('icpl-ichimaru-plus-style', $css);

}, 21);

/**
 * Block Editor: apply the same font.
 */
add_action('enqueue_block_editor_assets', function () {

    $font_key = (string) get_theme_mod('icpl_font_family', 'system_sans');

    $stacks = icpl_typography_get_font_stacks();
    $font_family = isset($stacks[$font_key]) ? $stacks[$font_key] : $stacks['system_sans'];

    wp_add_inline_style(
        'wp-edit-blocks',
        ':root{--icpl-font:' . $font_family . ';}.editor-styles-wrapper{font-family:var(--icpl-font);}'
    );

}, 20);