<?php
/**
 * Page Hero (Template Part)
 *
 * Responsibility:
 * - Render hero area for Pages.
 * - Supports per-page collage hero via post meta.
 * - Falls back to featured image hero (or no background).
 *
 * Expected post meta (per page):
 * - icpl_hero_collage_enable (bool/int "1")
 * - icpl_hero_collage_images (array of attachment IDs OR comma-separated string OR JSON)
 * - icpl_hero_collage_overlay (0-90) optional
 */

if (!defined('ABSPATH')) {
    exit;
}

$post_id = get_queried_object_id();
if (!$post_id) {
    $post_id = get_the_ID();
}

/**
 * Normalize collage images meta into an array of attachment IDs.
 *
 * @param mixed $raw
 * @return int[]
 */
if (!function_exists('icpl_normalize_collage_image_ids')) {
    function icpl_normalize_collage_image_ids($raw) {
        if (empty($raw)) {
            return [];
        }

        if (is_array($raw)) {
            $ids = $raw;
        } else {
            $trim = is_string($raw) ? trim($raw) : '';

            // JSON like: [1,2,3] or {"ids":[1,2,3]}
            if ($trim !== '' && ($trim[0] === '[' || $trim[0] === '{')) {
                $decoded = json_decode($trim, true);
                if (is_array($decoded)) {
                    if (isset($decoded['ids']) && is_array($decoded['ids'])) {
                        $ids = $decoded['ids'];
                    } else {
                        $ids = $decoded;
                    }
                } else {
                    $ids = [];
                }
            } else {
                // Comma-separated string: "1,2,3"
                $ids = is_string($raw)
                    ? preg_split('/\s*,\s*/', $raw, -1, PREG_SPLIT_NO_EMPTY)
                    : [];
            }
        }

        // Sanitize
        $ids = array_map('intval', (array)$ids);
        $ids = array_values(array_filter($ids, function($v){ return $v > 0; }));
        $ids = array_values(array_unique($ids));

        return $ids;
    }
}

$collage_enable = get_post_meta($post_id, 'icpl_hero_collage_enable', true);
$collage_enable = (string)$collage_enable === '1' || $collage_enable === 1 || $collage_enable === true;

$raw_ids   = get_post_meta($post_id, 'icpl_hero_collage_images', true);
$image_ids = icpl_normalize_collage_image_ids($raw_ids);

$use_collage = $collage_enable && !empty($image_ids);

// Overlay percent (optional; default 20)
$overlay = get_post_meta($post_id, 'icpl_hero_collage_overlay', true);
$overlay = is_numeric($overlay) ? (int)$overlay : 20;
$overlay = max(0, min(90, $overlay));

$overlay_style = get_post_meta($post_id, 'icpl_hero_overlay_style', true);
$overlay_style = ($overlay_style === 'dots') ? 'dots' : 'solid';
$overlay_class = ($overlay_style === 'dots')
    ? 'icpl-hero-overlay icpl-hero-overlay--dots'
    : 'icpl-hero-overlay';

$title = get_the_title($post_id);
$has_title = is_string($title) && $title !== '';

$vh = get_theme_mod('icpl_hero_height_vh', 22);
$vh = is_numeric($vh) ? (int)$vh : 22;
$vh = max(15, min(60, $vh));

$hero_classes = ['icpl-hero', 'icpl-page-hero'];
if ($use_collage) {
    $hero_classes[] = 'icpl-hero--collage';
} elseif (has_post_thumbnail($post_id)) {
    $hero_classes[] = 'icpl-hero--featured';
} else {
    $hero_classes[] = 'icpl-hero--plain';
}

$featured_url = '';
if (!$use_collage && has_post_thumbnail($post_id)) {
    $featured_url = get_the_post_thumbnail_url($post_id, 'full');
    if (!is_string($featured_url)) {
        $featured_url = '';
    }
}

$style_attr = 'min-height:' . esc_attr($vh) . 'vh;';
if (!$use_collage && $featured_url) {
    $style_attr .= 'background-image:url(' . esc_url($featured_url) . ');';
    $style_attr .= 'background-size:cover;background-position:center;';
}
?>

<section class="<?php echo esc_attr(implode(' ', $hero_classes)); ?>" style="<?php echo esc_attr($style_attr); ?>">

<?php if ($use_collage): ?>
    <div class="icpl-hero-bg icpl-hero-collage" aria-hidden="true" style="--icpl-collage-rows:3; grid-template-rows: repeat(var(--icpl-collage-rows), 1fr);">
        <?php
        // Randomize and limit to max 60 images for performance (JS will pick the exact needed count)
        if (count($image_ids) > 60) {
            shuffle($image_ids);
            $image_ids = array_slice($image_ids, 0, 60);
        }

        foreach ($image_ids as $attachment_id):
            $src = wp_get_attachment_image_url($attachment_id, 'large');
            if (!$src) {
                continue;
            }
            $alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
            $alt = is_string($alt) ? $alt : '';
        ?>
            <figure class="icpl-hero-collage__tile">
                <img
                    class="icpl-hero-collage__img"
                    src="<?php echo esc_url($src); ?>"
                    alt="<?php echo esc_attr($alt); ?>"
                    loading="lazy"
                    decoding="async"
                />
            </figure>
        <?php endforeach; ?>
    </div>

    <script>
      (function(){
        'use strict';

        function toInt(v, fallback){
          var n = parseInt(v, 10);
          return Number.isFinite(n) ? n : fallback;
        }

        function countGridTracks(trackList){
          if (!trackList) return 0;
          var m = trackList.match(/^repeat\((\d+),/);
          if (m) return toInt(m[1], 0);
          return trackList.trim().split(/\s+/).filter(Boolean).length;
        }

        function desiredRows(){
          var w = window.innerWidth || document.documentElement.clientWidth || 9999;
          if (w <= 480) return 5;
          if (w <= 768) return 4;
          return 3;
        }

        function syncCollage(){
          var grids = document.querySelectorAll('.icpl-page-hero.icpl-hero--collage .icpl-hero-bg.icpl-hero-collage');
          if (!grids.length) return;

          grids.forEach(function(grid){
            // Set rows responsively via CSS variable (no need to touch style.css)
            var rows = desiredRows();
            grid.style.setProperty('--icpl-collage-rows', String(rows));
            grid.style.gridTemplateRows = 'repeat(var(--icpl-collage-rows), 1fr)';

            var tiles = Array.prototype.slice.call(grid.querySelectorAll('.icpl-hero-collage__tile'));
            if (!tiles.length) return;

            tiles.forEach(function(t){ t.style.display = ''; });

            var cs = window.getComputedStyle(grid);
            var cols = countGridTracks(cs.gridTemplateColumns);
            if (!cols) cols = 3;

            var needed = Math.max(1, rows * cols);

            // If not enough tiles, clone cyclically until we have enough
            if (tiles.length < needed) {
              var i = 0;
              while (tiles.length < needed && i < 500) {
                var src = tiles[i % tiles.length];
                var clone = src.cloneNode(true);
                grid.appendChild(clone);
                tiles.push(clone);
                i++;
              }
            }

            // Hide extras beyond needed
            tiles.forEach(function(t, idx){
              if (idx >= needed) t.style.display = 'none';
            });
          });
        }

        function onResize(){
          window.clearTimeout(window.__icplCollageT);
          window.__icplCollageT = window.setTimeout(syncCollage, 120);
        }

        if (document.readyState === 'loading') {
          document.addEventListener('DOMContentLoaded', function(){
            syncCollage();
            window.addEventListener('resize', onResize);
          });
        } else {
          syncCollage();
          window.addEventListener('resize', onResize);
        }
      })();
    </script>

    <?php if ($overlay > 0): ?>
        <div class="<?php echo esc_attr($overlay_class); ?>" aria-hidden="true" style="opacity:<?php echo esc_attr($overlay / 100); ?>;"></div>
    <?php endif; ?>

<?php else: ?>

    <?php if ($featured_url): ?>
        <div class="<?php echo esc_attr($overlay_class); ?>" aria-hidden="true" style="opacity:0.25;"></div>
    <?php endif; ?>

<?php endif; ?>

    <div class="icpl-hero-inner">
        <div class="icpl-hero-content">
            <?php if ($has_title): ?>
                <h1 class="icpl-hero-title"><?php echo esc_html($title); ?></h1>
            <?php endif; ?>
        </div>
    </div>

</section>
